#include "fueldataprocess.h"
#include <cstdlib>

FuelDataProcess::FuelDataProcess(CInsim* inInSim, ConfigHandler* inCfgHdl)
{
	inSim = inInSim;
	cfgHdl = inCfgHdl;
	pthread_mutex_init(&mapMutex, NULL);

	consUnitsDesc[0] = "l/100 km";
	consUnitsDesc[1] = "MPG (UK)";
	consUnitsDesc[2] = "MPG (US)";

	consPHUnitsDesc[0] = "l/hr";
	consPHUnitsDesc[1] = "gal/hr (UK)";
	consPHUnitsDesc[2] = "gal/hr (US)";

	distUnitsDesc[0] = "km";
	distUnitsDesc[1] = "mi";
	distUnitsDesc[2] = "mi";

	units = cfgHdl->units;
	mode = cfgHdl->mode;

	memset(locCarID, 0, sizeof(locCarID));
  
	//Init the buttons we'll be sending through InSim
	//Button to display consumption
	memset(&is_dispOutputBtn, 0, sizeof(struct IS_BTN));
	is_dispOutputBtn.Type = ISP_BTN;
	is_dispOutputBtn.ReqI = 1;
	is_dispOutputBtn.UCID = 0;
	is_dispOutputBtn.ClickID = 239;
	is_dispOutputBtn.BStyle = ISB_LIGHT;
	is_dispOutputBtn.L = 1 + cfgHdl->xOffset;
	is_dispOutputBtn.T = 1 + cfgHdl->yOffset;
	is_dispOutputBtn.H = 5;
	is_dispOutputBtn.W = 35;

	//Toggle mode button
	memset(&is_switchModeBtn, 0, sizeof(struct IS_BTN));
	is_switchModeBtn.Type = ISP_BTN;
	is_switchModeBtn.ReqI = 1;
	is_switchModeBtn.UCID = 0;
	is_switchModeBtn.ClickID = EV_TOGGLEMODE;
	is_switchModeBtn.BStyle = ISB_LIGHT | ISB_CLICK;
	is_switchModeBtn.L = 1 + cfgHdl->xOffset;
	is_switchModeBtn.T = 7 + cfgHdl->yOffset;
	is_switchModeBtn.H = 5;
	is_switchModeBtn.W = 11;
	sprintf(is_switchModeBtn.Text, "%s", "Mode");

	//Reset counter button
	memset(&is_resetAverageBtn, 0, sizeof(struct IS_BTN));
	is_resetAverageBtn.Type = ISP_BTN;
	is_resetAverageBtn.ReqI = 1;
	is_resetAverageBtn.UCID = 0;
	is_resetAverageBtn.ClickID = EV_RESETAVG;
	is_resetAverageBtn.BStyle = ISB_LIGHT | ISB_CLICK;
	is_resetAverageBtn.L = 13 + cfgHdl->xOffset;
	is_resetAverageBtn.T = 7 + cfgHdl->yOffset;
	is_resetAverageBtn.H = 5;
	is_resetAverageBtn.W = 11;
	sprintf(is_resetAverageBtn.Text, "%s", "Reset");

	//Display config button
	memset(&is_dispConfigBtn, 0, sizeof(struct IS_BTN));
	is_dispConfigBtn.Type = ISP_BTN;
	is_dispConfigBtn.ReqI = 1;
	is_dispConfigBtn.UCID = 0;
	is_dispConfigBtn.ClickID = EV_DISPCONFIG;
	is_dispConfigBtn.BStyle = ISB_LIGHT | ISB_CLICK;
	is_dispConfigBtn.L = 25 + cfgHdl->xOffset;
	is_dispConfigBtn.T = 7 + cfgHdl->yOffset;
	is_dispConfigBtn.H = 5;
	is_dispConfigBtn.W = 11;
	sprintf(is_dispConfigBtn.Text, "%s", "Config");

	//Move up button
	memset(&is_upBtn, 0, sizeof(struct IS_BTN));
	is_upBtn.Type = ISP_BTN;
	is_upBtn.ReqI = 1;
	is_upBtn.UCID = 0;
	is_upBtn.ClickID = EV_UP;
	is_upBtn.BStyle = ISB_LIGHT | ISB_CLICK;
	is_upBtn.L = 1 + cfgHdl->xOffset;
	is_upBtn.T = 7 + cfgHdl->yOffset;
	is_upBtn.H = 5;
	is_upBtn.W = 5;
	sprintf(is_upBtn.Text, "%s", "U");

	//Move down button
	memset(&is_downBtn, 0, sizeof(struct IS_BTN));
	is_downBtn.Type = ISP_BTN;
	is_downBtn.ReqI = 1;
	is_downBtn.UCID = 0;
	is_downBtn.ClickID = EV_DOWN;
	is_downBtn.BStyle = ISB_LIGHT | ISB_CLICK;
	is_downBtn.L = 7 + cfgHdl->xOffset;
	is_downBtn.T = 7 + cfgHdl->yOffset;
	is_downBtn.H = 5;
	is_downBtn.W = 5;
	sprintf(is_downBtn.Text, "%s", "D");

	//Move left button
	memset(&is_leftBtn, 0, sizeof(struct IS_BTN));
	is_leftBtn.Type = ISP_BTN;
	is_leftBtn.ReqI = 1;
	is_leftBtn.UCID = 0;
	is_leftBtn.ClickID = EV_LEFT;
	is_leftBtn.BStyle = ISB_LIGHT | ISB_CLICK;
	is_leftBtn.L = 13 + cfgHdl->xOffset;
	is_leftBtn.T = 7 + cfgHdl->yOffset;
	is_leftBtn.H = 5;
	is_leftBtn.W = 5;
	sprintf(is_leftBtn.Text, "%s", "L");

	//Move right button
	memset(&is_rightBtn, 0, sizeof(struct IS_BTN));
	is_rightBtn.Type = ISP_BTN;
	is_rightBtn.ReqI = 1;
	is_rightBtn.UCID = 0;
	is_rightBtn.ClickID = EV_RIGHT;
	is_rightBtn.BStyle = ISB_LIGHT | ISB_CLICK;
	is_rightBtn.L = 19 + cfgHdl->xOffset;
	is_rightBtn.T = 7 + cfgHdl->yOffset;
	is_rightBtn.H = 5;
	is_rightBtn.W = 5;
	sprintf(is_rightBtn.Text, "%s", "R");

	//Toggle Units button
	memset(&is_switchUnitsBtn, 0, sizeof(struct IS_BTN));
	is_switchUnitsBtn.Type = ISP_BTN;
	is_switchUnitsBtn.ReqI = 1;
	is_switchUnitsBtn.UCID = 0;
	is_switchUnitsBtn.ClickID = EV_TOGGLEUNITS;
	is_switchUnitsBtn.BStyle = ISB_LIGHT | ISB_CLICK;
	is_switchUnitsBtn.L = 25 + cfgHdl->xOffset;
	is_switchUnitsBtn.T = 7 + cfgHdl->yOffset;
	is_switchUnitsBtn.H = 5;
	is_switchUnitsBtn.W = 11;
	sprintf(is_switchUnitsBtn.Text, "%s", "Units");
  
	//Go back button
	memset(&is_backBtn, 0, sizeof(struct IS_BTN));
	is_backBtn.Type = ISP_BTN;
	is_backBtn.ReqI = 1;
	is_backBtn.UCID = 0;
	is_backBtn.ClickID = EV_BACK;
	is_backBtn.BStyle = ISB_LIGHT | ISB_CLICK;
	is_backBtn.L = 25 + cfgHdl->xOffset;
	is_backBtn.T = 13 + cfgHdl->yOffset;
	is_backBtn.H = 5;
	is_backBtn.W = 11;
	sprintf(is_backBtn.Text, "%s", "Back");
  
	//Delete Mode button
	memset(&is_delModeBtn, 0, sizeof(struct IS_BFN));
	is_delModeBtn.Size = sizeof(struct IS_BFN);
	is_delModeBtn.Type= ISP_BFN;
	is_delModeBtn.ReqI = 1;
	is_delModeBtn.SubT = BFN_DEL_BTN;
	is_delModeBtn.UCID = 0;
	is_delModeBtn.ClickID = EV_TOGGLEMODE;
  
	//Delete Reset button
	memset(&is_delResetBtn, 0, sizeof(struct IS_BFN));
	is_delResetBtn.Size = sizeof(struct IS_BFN);
	is_delResetBtn.Type = ISP_BFN;
	is_delResetBtn.ReqI = 1;
	is_delResetBtn.SubT = BFN_DEL_BTN;
	is_delResetBtn.UCID = 0;
	is_delResetBtn.ClickID = EV_RESETAVG;

	//Delete Config button
	memset(&is_delConfigBtn, 0, sizeof(struct IS_BFN));
	is_delConfigBtn.Size = sizeof(struct IS_BFN);
	is_delConfigBtn.Type = ISP_BFN;
	is_delConfigBtn.ReqI = 1;
	is_delConfigBtn.SubT = BFN_DEL_BTN;
	is_delConfigBtn.UCID = 0;
	is_delConfigBtn.ClickID = EV_DISPCONFIG;

	//Delete all buttons
	memset(&is_delAllBtn, 0, sizeof(struct IS_BFN));
	is_delAllBtn.Size = sizeof(struct IS_BFN);
	is_delAllBtn.Type = ISP_BFN;
	is_delAllBtn.ReqI = 1;
	is_delAllBtn.SubT = BFN_CLEAR;
	is_delAllBtn.UCID = 0;
  
	//Display control buttons in LFS
	if(cfgHdl->dispInLFS)
		sendControlButtons();
	if(cfgHdl->dispInExt)
		extOut.displayWindow(this);
};


/**Compares the car ID we've revieved from LFS
 * to our database and returns tank capacity
 * of the car that's being used
 */
float FuelDataProcess::getTankVolume(char* carID)
{
	if(strcmp(carID, "UF1") == 0) return 35.0f;
	else if(strcmp(carID, "XFG") == 0) return 45.0f;
	else if(strcmp(carID, "XRG") == 0) return 65.0f;
	else if(strcmp(carID, "LX4") == 0) return 40.0f;
	else if(strcmp(carID, "LX6") == 0) return 40.0f;
	else if(strcmp(carID, "RB4") == 0) return 75.0f;
	else if(strcmp(carID, "FXO") == 0) return 75.0f;
	else if(strcmp(carID, "XRT") == 0) return 75.0f;
	else if(strcmp(carID, "RAC") == 0) return 42.0f;
	else if(strcmp(carID, "FZ5") == 0) return 90.0f;
	else if(strcmp(carID, "RAC") == 0) return 42.0f;
	else if(strcmp(carID, "UFR") == 0) return 60.0f;
	else if(strcmp(carID, "XFR") == 0) return 70.0f;
	else if(strcmp(carID, "RAC") == 0) return 42.0f;
	else if(strcmp(carID, "FXR") == 0) return 100.0f;
	else if(strcmp(carID, "XRR") == 0) return 100.0f;
	else if(strcmp(carID, "FZR") == 0) return 100.0f;
	else if(strcmp(carID, "MRT") == 0) return 20.0f;
	else if(strcmp(carID, "FBM") == 0) return 42.0f;
	else if(strcmp(carID, "FOX") == 0) return 38.0f;
	else if(strcmp(carID, "FO8") == 0) return 125.0f;
	else if(strcmp(carID, "BF1") == 0) return 95.0f;
	else return -1.0f;
}

/** Gets data from the recieved OutGauge packet
* and calculates instant consumption
in litres per 100 km
*/
void FuelDataProcess::process(const float fuel, const float speed, const unsigned int time, char* const carID, const byte plid)
{
	//Make a local copy of carID in case we receive new OutGauge packet before this one gets processed
	strcpy(locCarID, carID);
	
	activePlid = plid;
	CarFuelData* fd;
	
	pthread_mutex_lock(&mapMutex);			//Do not allow data to be resetted when we're updating consumption data
	it = carFD.find(plid);
	if(it == carFD.end()) {					//No data for this player yet
		fd = new CarFuelData();
		carFD[plid] = fd;
	} else									//We already have data for this car
		fd = it->second;
  
	if(fd->firstData) {						//Is this the first data we recieved since last reset?
		fd->tankVolume = getTankVolume(locCarID);
		strcpy(fd->carID, locCarID);
		fd->prevTime = time;
		fd->prevFuel = fuel;
		fd->prevSpeed = speed;
		fd->firstData = false;
		pthread_mutex_unlock(&mapMutex);
		return;
	}
	
	if(strcmp(locCarID, fd->carID) != 0) {		//Player's car changed, reset counter
		pthread_mutex_unlock(&mapMutex);	
		reInitPlr(plid);
		return;
	}

	deltaTime = time - fd->prevTime;
	if(deltaTime > cfgHdl->updateSpeed) {							//Update fuel consumption info
		deltaTime /= 1000.0f;										//Convert deltaTime to seconds
		deltaFuel = (fd->prevFuel - fuel) * fd->tankVolume;			//How much fuel was burnt since last update
		if(speed > 1.0f) {
			deltaS = (speed + fd->prevSpeed) / 2.0f * deltaTime;	//How far the car traveled since last update (approximation)
			fd->instantCons = (100000.0f / deltaS) * deltaFuel;		//How much fuel would be needed to cover 100km distance with present consumption
      
			fd->totalFuelUsed += deltaFuel;							//Total amount of fuel burnt while the car was moving
		} else {
			deltaS = 0.0f;
			fd->instantCons = -1.0f;
		}
    
		fd->totalDistance += deltaS;

		//Update the info about last known time and fuel level
		fd->prevTime = time;
		fd->prevFuel = fuel;
		fd->prevSpeed = speed;
		fd->leftInTank = fuel * fd->tankVolume;

		//Calculate fuel per hour
		fd->fuelPerHour = (3600.0f / (deltaTime)) * deltaFuel;
    
		//Calculate average consumption in litres per 100 km
		if(fd->totalDistance > 1.0f) {
			fd->averageCons = (fd->totalFuelUsed * 100000.0f) / fd->totalDistance;
			//Calculate projected range
			fd->range = fd->leftInTank / (fd->averageCons / 100.0f);
		}
		displayData(fd);
	}
	pthread_mutex_unlock(&mapMutex);
}

/** Update is_dispOutputBtn packet
*   and send it to LFS.*/
void FuelDataProcess::displayData(CarFuelData* const fd)
{
	switch(mode) {
		case MODE_INSTANT:
			if(fd->instantCons > 0.0f)
				sprintf(is_dispOutputBtn.Text, "Instant: %.f %s", convertCons(fd->instantCons), consUnitsDesc[units]);
			else if(fd->instantCons == 0.0f) {
				switch(units) {
					case UNITS_METRIC:
						sprintf(is_dispOutputBtn.Text, "Instant: %.f %s", 0.0f, consUnitsDesc[units]);
					default:
						sprintf(is_dispOutputBtn.Text, "Instant: %.f %s", 999.0f, consUnitsDesc[units]);
				}
			} else
				sprintf(is_dispOutputBtn.Text, "Instant: --.- %s", consUnitsDesc[units]);	  
		break;
	
		case MODE_AVERAGE:
			if(fd->totalDistance > 1.0f && fd->averageCons > 0.0f)
				sprintf(is_dispOutputBtn.Text, "Average: %.2f %s", convertCons(fd->averageCons), consUnitsDesc[units]);
			else if(fd->totalDistance > 1.0f && fd->averageCons == 0.0f) {
				switch(units) {
					case UNITS_METRIC:
						sprintf(is_dispOutputBtn.Text, "Instant: %.f %s", 0.0f, consUnitsDesc[units]);
					default:
						sprintf(is_dispOutputBtn.Text, "Instant: %.f %s", 999.0f, consUnitsDesc[units]);
				}
			} else
				sprintf(is_dispOutputBtn.Text, "Average: --.- %s", consUnitsDesc[units]);
		break;

		case MODE_PER_HOUR:
			sprintf(is_dispOutputBtn.Text, "Instant: %.2f %s", convertVolume(fd->fuelPerHour), consPHUnitsDesc[units]);
		break;
	
		case MODE_RANGE:
			if(fd->averageCons > 0.0f)
				sprintf(is_dispOutputBtn.Text, "Range: %.0f %s", convertDistance(fd->range), distUnitsDesc[units]);
			else
				sprintf(is_dispOutputBtn.Text, "Range: --- %s", distUnitsDesc[units]);
		break;
	}
	/*
	sprintf(is_dispOutputBtn.Text, "--- NO DATA ---");
	*/

	if(cfgHdl->dispInLFS)				//Display output in LFS using IS_BTN
		inSim->send_packet(&is_dispOutputBtn);
	if(cfgHdl->dispInExt) {				//Display output in external window
		if(extOut.checkState()) {
			//Copy the output text out of the InSim packet		
			memset(extOut.outputText, 0, sizeof(extOut.outputText));
			memcpy(extOut.outputText, is_dispOutputBtn.Text, strlen(is_dispOutputBtn.Text));
			extOut.updateData();
		}
	}
}

/** Converts consumption in litres/100 km to
* MPG if Imperial or US units are seleceted.
*/
const float FuelDataProcess::convertCons(const float inCons)
{
	float conv;
	switch(units) {
		case UNITS_IMPERIAL:
			conv = 282.54f/inCons;
				return  conv > 999.0f ? 999.0f : conv;
		case UNITS_US:
			conv = 253.26f/inCons;
				return  conv > 999.0f ? 999.0f : conv;
		default:
			return inCons;
	}
}

/** Converts fuel volume to either Imperial
* or US gallon if needed.
*/
const float FuelDataProcess::convertVolume(const float fuelPerHour)
{
	switch(units)
	{
	case UNITS_IMPERIAL:
		return fuelPerHour/4.5461f;
	case UNITS_US:
		return fuelPerHour/3.7854f;
	default:
		return fuelPerHour;
	}
}

/**Converts distance in kilometers to miles
* if other than metric units are selected.
*/
const float FuelDataProcess::convertDistance(const float range)
{
	if(units > UNITS_METRIC)
		return range/1.609f;
	else
		return range;
}

/** User clicked "Units" button in LFS,
* so select next unit system in the list
*/
void FuelDataProcess::toggleUnits()
{
	if(units < 2)
		units++;
	else
		units = UNITS_METRIC;
  
	cfgHdl->units = units;
};

/** User clicked "Mode" button in LFS,
* so select next mode.
*/
void FuelDataProcess::toggleMode()
{
	if(mode < 3)
		mode++;
	else
		mode = MODE_INSTANT;
  
	cfgHdl->mode = mode;
};

/** User clicked a button in LFS,
* so let's find out which one
*/
void FuelDataProcess::incomingEvent(byte clickID)
{
	if(clickID == EV_TOGGLEUNITS)	{ //Toggle units
		toggleUnits();
	}
	else if(clickID == EV_TOGGLEMODE) {	//Toggle modes
		toggleMode();
	}
	else if(clickID == EV_RESETAVG) {	//Reset average consumption for player whose packet we got the last
		reInitPlr(activePlid);
	}
	else if(clickID == EV_DISPCONFIG) {	//Display configuration buttons
	//Delete currently displayed control buttons
	inSim->send_packet(&is_delModeBtn);
	inSim->send_packet(&is_delResetBtn);
	inSim->send_packet(&is_delConfigBtn);
    
	//Display config buttons
	sendConfigButtons();
	}
	else if(clickID == EV_UP) {		//Move interface up
		cfgHdl->yOffset--;
		recalcButtonsPos();
		inSim->send_packet(&is_delAllBtn);
		sendConfigButtons();
	}
	else if(clickID == EV_DOWN) {		//Move interface down
		cfgHdl->yOffset++;
		recalcButtonsPos();
		inSim->send_packet(&is_delAllBtn);
		sendConfigButtons();
	}
	else if(clickID == EV_LEFT) {		//Move interface left
		cfgHdl->xOffset--;
		recalcButtonsPos();
		inSim->send_packet(&is_delAllBtn);
		sendConfigButtons();
	}
	else if(clickID == EV_RIGHT) {	//Move interface down
		cfgHdl->xOffset++;
		recalcButtonsPos();
		inSim->send_packet(&is_delAllBtn);
		sendConfigButtons();
	}
	else if(clickID == EV_BACK) {		//Close config and return to consumption display
		cfgHdl->saveConfigFile();
		inSim->send_packet(&is_delAllBtn);
		sendControlButtons();
	}
};

void FuelDataProcess::sendConfigButtons()
{
	inSim->send_packet(&is_upBtn);
	inSim->send_packet(&is_downBtn);
	inSim->send_packet(&is_leftBtn);
	inSim->send_packet(&is_rightBtn);
	inSim->send_packet(&is_switchUnitsBtn);
	inSim->send_packet(&is_backBtn);
}

void FuelDataProcess::sendControlButtons()
{
	inSim->send_packet(&is_switchModeBtn);
	inSim->send_packet(&is_resetAverageBtn);
	inSim->send_packet(&is_dispConfigBtn);
}

void FuelDataProcess::recalcButtonsPos()
{
	is_dispOutputBtn.L = 1 + cfgHdl->xOffset;
	is_dispOutputBtn.T = 1 + cfgHdl->yOffset;
  
	is_switchModeBtn.L = 1 + cfgHdl->xOffset;
	is_switchModeBtn.T = 7 + cfgHdl->yOffset;
  
	is_resetAverageBtn.L = 13 + cfgHdl->xOffset;
	is_resetAverageBtn.T = 7 + cfgHdl->yOffset;
  
	is_dispConfigBtn.L = 25 + cfgHdl->xOffset;
	is_dispConfigBtn.T = 7 + cfgHdl->yOffset;
  
	is_upBtn.L = 1 + cfgHdl->xOffset;
	is_upBtn.T = 7 + cfgHdl->yOffset;
  
	is_downBtn.L = 7 + cfgHdl->xOffset;
	is_downBtn.T = 7 + cfgHdl->yOffset;
  
	is_leftBtn.L = 13 + cfgHdl->xOffset;
	is_leftBtn.T = 7 + cfgHdl->yOffset;
  
	is_rightBtn.L = 19 + cfgHdl->xOffset;
	is_rightBtn.T = 7 + cfgHdl->yOffset;
  
	is_switchUnitsBtn.L = 25 + cfgHdl->xOffset;
	is_switchUnitsBtn.T = 7 + cfgHdl->yOffset;
  
	is_backBtn.L = 25 + cfgHdl->xOffset;
	is_backBtn.T = 13 + cfgHdl->yOffset;
}

/** Counter has to be reseted (car changed, user requested so), so make sure
*   we reset it upon next update
*/
void FuelDataProcess::reInitPlr(byte plid)
{
	pthread_mutex_lock(&mapMutex);				//Prevent manipulation with the map when we're resetting consumption data
	std::map<byte, CarFuelData*>::iterator it = carFD.find(plid);
	if(it != carFD.end()) {
		delete it->second;
		carFD.erase(it);
	}
	pthread_mutex_unlock(&mapMutex);
}

/** Race restart, delete consumption data for all cars */
void FuelDataProcess::reInitFull()
{
	pthread_mutex_lock(&mapMutex);
	std::map<byte, CarFuelData*>::iterator it;
	for(it = carFD.begin(); it != carFD.end(); it++) {
		delete it->second;
		carFD.erase(it);
	}
	pthread_mutex_unlock(&mapMutex);
}

FuelDataProcess::~FuelDataProcess()
{
	pthread_mutex_destroy(&mapMutex);
}