﻿using System;
using System.Drawing;
using System.Windows.Forms;
using LfsPoints.Commands;
using LfsPoints.Projects;
using LfsPoints.Properties;

namespace LfsPoints {
    public class AllocationTab : TabPage {
        private Allocation _allocation;
        private TableLayoutPanel _allocationTable;
        private Button _addPointsButton;
        private Button _removePointsButton;
        private FlowLayoutPanel _allocationFlowPanel;
        private DataGridViewTextBoxColumn _positionColumn;
        private DataGridViewTextBoxColumn _pointsColumn;
        private DataGridView _pointsDataGridView;
        private Label _winnerLabel;
        private NumericUpDown _winnerNumericUpDown;
        private Label _polePositionLabel;
        private NumericUpDown _polePositionNumericUpDown;
        private Label _fastestLapLabel;
        private NumericUpDown _fastestLapNumericUpDown;
        private Label _highestClimbLabel;
        private NumericUpDown _highestClimbNumericUpDown;
        private Label _percentCompleteLabel;
        private NumericUpDown _percentCompleteNumericUpDown;
        private CheckBox _omitDisqualifiedCheckBox;
        private int _currentPosition;

        protected static IWorkbench AppWorkbench {
            get { return (IWorkbench)Application.OpenForms[0]; }
        }

        public AllocationTab(Allocation allocation)
            : base(allocation.Name) {
            InitializeTab();

            Tag = allocation;
            allocation.Tag = this;
            _allocation = allocation;

            // Populate tab.
            foreach (int points in _allocation.Points) {
                _pointsDataGridView.Rows.Add(++_currentPosition, points);
            }
            _winnerNumericUpDown.Value = _allocation.Winner;
            _polePositionNumericUpDown.Value = _allocation.PolePosition;
            _fastestLapNumericUpDown.Value = _allocation.FastestLap;
            _highestClimbNumericUpDown.Value = _allocation.HighestClimb;
            _percentCompleteNumericUpDown.Value = _allocation.PercentCompleted;
            _omitDisqualifiedCheckBox.Checked = _allocation.OmitDisqualified;

            // Hook events.
            _winnerNumericUpDown.ValueChanged += new EventHandler(_winnerNumericUpDown_ValueChanged);
            _polePositionNumericUpDown.ValueChanged += new EventHandler(_polePositionNumericUpDown_ValueChanged);
            _fastestLapNumericUpDown.ValueChanged += new EventHandler(_fastestLapNumericUpDown_ValueChanged);
            _highestClimbNumericUpDown.ValueChanged += new EventHandler(_highestClimbNumericUpDown_ValueChanged);
            _percentCompleteNumericUpDown.ValueChanged += new EventHandler(_percentCompleteNumericUpDown_ValueChanged);
            _omitDisqualifiedCheckBox.CheckedChanged += new EventHandler(_omitDisqualifiedCheckBox_CheckedChanged);
        }

        private void InitializeTab() {
            SuspendLayout();
            UseVisualStyleBackColor = true;

            _allocationTable = new TableLayoutPanel();
            _allocationTable.Dock = DockStyle.Fill;
            _allocationTable.ColumnCount = 2;
            _allocationTable.ColumnStyles.Add(new ColumnStyle(SizeType.AutoSize));
            _allocationTable.ColumnStyles.Add(new ColumnStyle(SizeType.Percent, 100f));
            _allocationTable.RowCount = 8;
            _allocationTable.RowStyles.Add(new RowStyle(SizeType.AutoSize));
            _allocationTable.RowStyles.Add(new RowStyle(SizeType.Percent, 100f));
            _allocationTable.RowStyles.Add(new RowStyle(SizeType.AutoSize));
            _allocationTable.RowStyles.Add(new RowStyle(SizeType.AutoSize));
            _allocationTable.RowStyles.Add(new RowStyle(SizeType.AutoSize));
            _allocationTable.RowStyles.Add(new RowStyle(SizeType.AutoSize));
            _allocationTable.RowStyles.Add(new RowStyle(SizeType.AutoSize));
            _allocationTable.RowStyles.Add(new RowStyle(SizeType.AutoSize));

            _addPointsButton = new Button();
            _addPointsButton.AutoSize = true;
            _addPointsButton.AutoSizeMode = AutoSizeMode.GrowAndShrink;
            _addPointsButton.Image = Resources.add;
            _addPointsButton.Margin = new Padding(3, 3, 0, 0);
            _addPointsButton.TabIndex = 0;
            _addPointsButton.Click += new System.EventHandler(_addPointsButton_Click);

            _removePointsButton = new Button();
            _removePointsButton.AutoSize = true;
            _removePointsButton.AutoSizeMode = AutoSizeMode.GrowAndShrink;
            _removePointsButton.Image = Resources.delete;
            _removePointsButton.Margin = new Padding(3, 3, 3, 0);
            _removePointsButton.TabIndex = 1;
            _removePointsButton.Click += new System.EventHandler(_removePointsButton_Click);

            _allocationFlowPanel = new FlowLayoutPanel();
            _allocationFlowPanel.Margin = new Padding(0);
            _allocationFlowPanel.FlowDirection = FlowDirection.RightToLeft;
            _allocationFlowPanel.Dock = DockStyle.Fill;
            _allocationFlowPanel.AutoSize = true;
            _allocationFlowPanel.AutoSizeMode = AutoSizeMode.GrowAndShrink;
            _allocationFlowPanel.Controls.AddRange(new Control[] {
                _removePointsButton,
                _addPointsButton,
            });
            _allocationTable.Controls.Add(_allocationFlowPanel, 0, 0);
            _allocationTable.SetColumnSpan(_allocationFlowPanel, 2);

            _positionColumn = new DataGridViewTextBoxColumn();
            _positionColumn.HeaderText = StringResources.AllocationTabPositionColumnText;
            _positionColumn.ReadOnly = true;
            _positionColumn.SortMode = DataGridViewColumnSortMode.NotSortable;
            _pointsColumn = new DataGridViewTextBoxColumn();
            _pointsColumn.HeaderText = StringResources.AllocationTabPointsColumnText;
            _pointsColumn.SortMode = DataGridViewColumnSortMode.NotSortable;

            _pointsDataGridView = new DataGridView();
            _pointsDataGridView.Dock = DockStyle.Fill;
            _pointsDataGridView.AllowUserToAddRows = false;
            _pointsDataGridView.AllowUserToDeleteRows = false;
            _pointsDataGridView.AllowUserToOrderColumns = false;
            _pointsDataGridView.AllowUserToResizeColumns = false;
            _pointsDataGridView.AllowUserToResizeRows = false;
            _pointsDataGridView.AutoSizeColumnsMode = DataGridViewAutoSizeColumnsMode.Fill;
            _pointsDataGridView.Columns.AddRange(new DataGridViewColumn[] {
                _positionColumn,
                _pointsColumn,
            });
            _pointsDataGridView.EditMode = DataGridViewEditMode.EditOnEnter;
            _pointsDataGridView.RowHeadersVisible = false;
            _pointsDataGridView.SelectionMode = DataGridViewSelectionMode.FullRowSelect;
            _pointsDataGridView.ColumnHeadersHeightSizeMode = DataGridViewColumnHeadersHeightSizeMode.DisableResizing;
            _pointsDataGridView.MultiSelect = false;
            _pointsDataGridView.BackgroundColor = SystemColors.Window;
            _pointsDataGridView.BorderStyle = BorderStyle.None;
            _pointsDataGridView.GridColor = SystemColors.ControlLight;
            _pointsDataGridView.CellValidating += new DataGridViewCellValidatingEventHandler(_pointsDataGridView_CellValidating);
            _pointsDataGridView.CellValueChanged += new DataGridViewCellEventHandler(_pointsDataGridView_CellValueChanged);
            _allocationTable.Controls.Add(_pointsDataGridView, 0, 1);
            _allocationTable.SetColumnSpan(_pointsDataGridView, 2);

            _winnerLabel = new Label();
            _winnerLabel.Text = StringResources.AllocationTabWinnerLabelText;
            _winnerLabel.Anchor = AnchorStyles.Left;
            _winnerLabel.AutoSize = true;
            _allocationTable.Controls.Add(_winnerLabel, 0, 2);
            _winnerNumericUpDown = new NumericUpDown();
            _winnerNumericUpDown.Dock = DockStyle.Fill;
            _winnerNumericUpDown.Minimum = Int32.MinValue;
            _winnerNumericUpDown.Maximum = Int32.MaxValue;
            _allocationTable.Controls.Add(_winnerNumericUpDown, 1, 2);

            _polePositionLabel = new Label();
            _polePositionLabel.Text = StringResources.AllocationTabPolePositionLabelText;
            _polePositionLabel.Anchor = AnchorStyles.Left;
            _polePositionLabel.AutoSize = true;
            _allocationTable.Controls.Add(_polePositionLabel, 0, 3);
            _polePositionNumericUpDown = new NumericUpDown();
            _polePositionNumericUpDown.Dock = DockStyle.Fill;
            _polePositionNumericUpDown.Minimum = Int32.MinValue;
            _polePositionNumericUpDown.Maximum = Int32.MaxValue;
            _allocationTable.Controls.Add(_polePositionNumericUpDown, 1, 3);

            _fastestLapLabel = new Label();
            _fastestLapLabel.Text = StringResources.AllocationTabFastestLapLabelText;
            _fastestLapLabel.Anchor = AnchorStyles.Left;
            _fastestLapLabel.AutoSize = true;
            _allocationTable.Controls.Add(_fastestLapLabel, 0, 4);
            _fastestLapNumericUpDown = new NumericUpDown();
            _fastestLapNumericUpDown.Dock = DockStyle.Fill;
            _fastestLapNumericUpDown.Minimum = Int32.MinValue;
            _fastestLapNumericUpDown.Maximum = Int32.MaxValue;
            _allocationTable.Controls.Add(_fastestLapNumericUpDown, 1, 4);

            _highestClimbLabel = new Label();
            _highestClimbLabel.Text = StringResources.AllocationTabHighestClimbLabelText;
            _highestClimbLabel.Anchor = AnchorStyles.Left;
            _highestClimbLabel.AutoSize = true;
            _allocationTable.Controls.Add(_highestClimbLabel, 0, 5);
            _highestClimbNumericUpDown = new NumericUpDown();
            _highestClimbNumericUpDown.Dock = DockStyle.Fill;
            _highestClimbNumericUpDown.Minimum = Int32.MinValue;
            _highestClimbNumericUpDown.Maximum = Int32.MaxValue;
            _allocationTable.Controls.Add(_highestClimbNumericUpDown, 1, 5);

            _percentCompleteLabel = new Label();
            _percentCompleteLabel.Text = StringResources.AllocationTabPercentCompletedLabelText;
            _percentCompleteLabel.Anchor = AnchorStyles.Left;
            _percentCompleteLabel.AutoSize = true;
            _allocationTable.Controls.Add(_percentCompleteLabel, 0, 6);
            _percentCompleteNumericUpDown = new NumericUpDown();
            _percentCompleteNumericUpDown.Dock = DockStyle.Fill;
            _percentCompleteNumericUpDown.Minimum = 0;
            _percentCompleteNumericUpDown.Maximum = 100;
            _allocationTable.Controls.Add(_percentCompleteNumericUpDown, 1, 6);

            _omitDisqualifiedCheckBox = new CheckBox();
            _omitDisqualifiedCheckBox.AutoSize = true;
            _omitDisqualifiedCheckBox.Text = StringResources.AllocationTabOmitDisqualifiedCheckBoxText;
            _omitDisqualifiedCheckBox.Margin = new Padding(6, 3, 3, 3);
            _allocationTable.Controls.Add(_omitDisqualifiedCheckBox, 0, 7);
            _allocationTable.SetColumnSpan(_omitDisqualifiedCheckBox, 2);

            Controls.Add(_allocationTable);
            ResumeLayout();
        }

        private void _addPointsButton_Click(object sender, EventArgs e) {
            int index = _pointsDataGridView.Rows.Add(++_currentPosition, 0);
            _pointsDataGridView.Rows[index].Selected = true;
            _pointsDataGridView.FirstDisplayedScrollingRowIndex = index;

            AppWorkbench.ExecuteCommand(new AddPointsCommand(_allocation));
        }

        private void _removePointsButton_Click(object sender, EventArgs e) {
            int index = _pointsDataGridView.Rows.GetLastRow();
            if (index > -1) {
                _pointsDataGridView.Rows.RemoveAt(--_currentPosition);
                AppWorkbench.ExecuteCommand(new RemovePointsCommand(_allocation));
            }
        }

        private void _pointsDataGridView_CellValidating(object sender, DataGridViewCellValidatingEventArgs e) {
            int temp;
            e.Cancel = !Int32.TryParse((string)e.FormattedValue, out temp);
        }

        private void _pointsDataGridView_CellValueChanged(object sender, DataGridViewCellEventArgs e) {
            int points = Convert.ToInt32(_pointsDataGridView.Rows[e.RowIndex].Cells[e.ColumnIndex].Value);

            AppWorkbench.ExecuteCommand(new PointsChangedCommand(_allocation, e.RowIndex, points));
        }

        private void _winnerNumericUpDown_ValueChanged(object sender, EventArgs e) {
            AppWorkbench.ExecuteCommand(new WinnerPointsChangedCommand(_allocation, (int)_winnerNumericUpDown.Value));
        }

        private void _polePositionNumericUpDown_ValueChanged(object sender, EventArgs e) {
            AppWorkbench.ExecuteCommand(new PolePositionPointsChangedCommand(_allocation, (int)_polePositionNumericUpDown.Value));
        }

        private void _fastestLapNumericUpDown_ValueChanged(object sender, EventArgs e) {
            AppWorkbench.ExecuteCommand(new FastestLapPointsChangedCommand(_allocation, (int)_fastestLapNumericUpDown.Value));
        }

        private void _highestClimbNumericUpDown_ValueChanged(object sender, EventArgs e) {
            AppWorkbench.ExecuteCommand(new HighestClimbPointsChangedCommand(_allocation, (int)_highestClimbNumericUpDown.Value));
        }

        private void _percentCompleteNumericUpDown_ValueChanged(object sender, EventArgs e) {
            AppWorkbench.ExecuteCommand(new PercentCompletedChangedCommand(_allocation, (int)_percentCompleteNumericUpDown.Value));
        }

        private void _omitDisqualifiedCheckBox_CheckedChanged(object sender, EventArgs e) {
            AppWorkbench.ExecuteCommand(new OmitDisqualifiedPointsChangedCommand(_allocation, _omitDisqualifiedCheckBox.Checked));
        }
    }
}
