﻿using System;
using System.IO;
using System.Windows.Forms;
using LfsPoints.Projects;
using LfsPoints.Properties;

namespace LfsPoints.Commands {
    public class ExportProjectCommand : ICommand {
        public bool Execute(IWorkbench workbench) {
            Project project = Project.Current;

            using (SaveFileDialog dialog = new SaveFileDialog()) {
                dialog.Title = StringResources.ExportProjectDialogTitle;
                dialog.Filter = StringResources.ExportProjectFilter;
                dialog.AddExtension = true;
                dialog.OverwritePrompt = true;
                dialog.FileName = project.Name;
                dialog.InitialDirectory = Settings.Default.ExportDirectory;
                dialog.FilterIndex = Settings.Default.ExportFilter;

                if (DialogHelper.ShowDialog(dialog)) {
                    try {
                        workbench.Cursor = Cursors.WaitCursor;

                        string ext = Path.GetExtension(dialog.FileName);

                        IExporter exporter = ExporterFactory.GetExporter(ext);

                        if (exporter == null) {
                            DialogHelper.ShowError(
                                StringResources.ExportProjectUnsupportedErrorDialogTitle,
                                StringResources.ExportProjectUnsupportedErrorDialogMessage,
                                ext);
                        }
                        else {
                            exporter.Export(dialog.FileName, project);

                            Settings.Default.ExportFilter = dialog.FilterIndex;
                            Settings.Default.ExportDirectory = Path.GetDirectoryName(dialog.FileName);

                            DialogHelper.ShowInfo(
                                StringResources.ExportProjectSuccessDialogTitle,
                                StringResources.ExportProjectSuccessDialogMessage,
                                project.Name);
                        }
                    }
#if !DEBUG
                    catch (Exception ex) {
                        if (ex.InnerException == null) {
                            DialogHelper.ShowError(
                                StringResources.ExportProjectErrorDialogTitle,
                                StringResources.ExportProjectErrorDialogMessage,
                                ex.Message);
                        }
                        else {
                            // XSL parse errors are inside InnerException.
                            DialogHelper.ShowError(
                                StringResources.ExportProjectErrorDialogTitle,
                                StringResources.ExportProjectErrorDialogMessage,
                                ex.InnerException.Message);
                        }
                    }
#endif
                    finally {
                        workbench.Cursor = null;
                    }
                }
            }

            return false;
        }
    }
}
