﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Xml.Serialization;

namespace LfsPoints.Projects {
    [Serializable]
    [DebuggerDisplay("Name = {Name}")]
    public class Project {
        private static Project _current;
        private string _defaultName;

        public static Project Current {
            get { return _current; }
            set { _current = value; }
        }

        public DisplayNameDictionary<string, string> DisplayNames { get; set; }
        public List<Allocation> Allocations { get; set; }
        public List<Replay> Replays { get; set; }

        [XmlIgnore]
        public bool IsDirty { get; set; }

        [XmlIgnore]
        public string FileName { get; set; }

        [XmlIgnore]
        public string Directory {
            get { return Path.GetDirectoryName(FileName); }
        }

        [XmlIgnore]
        public bool IsSavable {
            get { return !String.IsNullOrEmpty(FileName); }
        }

        [XmlIgnore]
        public string Name {
            get {
                if (IsSavable) {
                    return Path.GetFileNameWithoutExtension(FileName);
                }
                return _defaultName;
            }
        }

        [XmlIgnore]
        public IEnumerable<Standing> Standings {
            get { return GetStandings(); }
        }

        public Project() {
            DisplayNames = new DisplayNameDictionary<string, string>();
            Allocations = new List<Allocation>();
            Replays = new List<Replay>();
        }

        public Project(string defaultName)
            : this() {
            _defaultName = defaultName;
        }

        public string GetDisplayName(string userName) {
            return DisplayNames[userName];
        }

        public void SetDisplayName(string userName, string displayName) {
            DisplayNames[userName] = displayName;
        }

        public void TrySetDisplayName(string userName, string displayName) {
            if (!DisplayNames.ContainsKey(userName)) {
                DisplayNames[userName] = displayName;
            }
        }

        private IEnumerable<Standing> GetStandings() {
            Dictionary<string, Standing> standings = new Dictionary<string, Standing>();

            var results = from r in Replays
                          from s in r.Results
                          select s;

            foreach (Result result in results) {
                int points = result.Points;

                Standing standing;
                if (standings.TryGetValue(result.UserName, out standing)) {
                    standing.Points += points;
                }
                else {
                    standing = new Standing(result.UserName, points);
                    standings.Add(result.UserName, standing);
                }

                if (result.Position == 1) {
                    standing.Wins++;
                }
                else if (result.Position >= 1 && result.Position <= 3) {
                    standing.Podiums++;
                }
            }

            return standings.Values
                .OrderByDescending(s => s.Points)
                .ThenByDescending(s => s.Wins)
                .ThenByDescending(s => s.Podiums)
                .Select((s, i) => {
                    s.Position = i + 1;
                    return s;
                });
        }

        public static void Save() {
            Save(Current.FileName);
        }

        public static void Save(string fileName) {
            SerializationHelper.Serialize(fileName, Current);
            Current.FileName = fileName;
            Current.IsDirty = false;
        }

        public static Project Load(string fileName) {
            Project project = SerializationHelper.Deserialize<Project>(fileName);
            project.FileName = fileName;
            project.IsDirty = false;

            UpdateReferences(project);

            return project;
        }

        private static void UpdateReferences(Project project) {
            foreach (Replay replay in project.Replays) {
                replay.Allocation = project.Allocations.Single(a => a.AllocationId == replay.AllocationId);
                replay.ReorderResults();

                foreach (Result result in replay.Results) {
                    result.Replay = project.Replays.Single(r => r.ReplayId == result.ReplayId);
                }
            }
        }
    }
}
