﻿using System;
using System.Collections.Generic;
using System.Linq;

namespace LfsPoints.Projects {
    public static class ReplayHelper {
        public static string GetTrack(Replay replay) {
            return TrackHelper.GetFullTrackName(replay.TrackName);
        }

        public static string GetRaceMode(Replay replay) {
            switch (replay.RaceMode) {
                case RaceMode.Practice:
                    return StringResources.ResultInfoRaceModePractice;
                case RaceMode.Qualifying:
                    return StringResources.ResultInfoRaceModeQualifying;
                default:
                    return StringResources.ResultInfoRaceModeRace;
            }
        }

        public static string GetLaps(Replay replay) {
            int laps = replay.Laps;
            if (laps > 0) {
                return laps.ToString();
            }
            else if (replay.Hours > 0) {
                return String.Format(StringResources.ResultLapsHours, replay.Hours);
            }
            return String.Empty;
        }

        public static string GetCar(Result result) {
            return CarHelper.GetFullCarName(result.CarName);
        }

        public static string GetGrid(Result result) {
            if (result.IsQualifying) {
                return String.Empty;
            }

            if (result.StartPosition > 0) {
                int climb = result.Climb;

                if (climb > 0) {
                    return String.Format(StringResources.ResultInfoGridClimb, result.StartPosition, climb);
                }
                else if (climb < 0) {
                    return String.Format(StringResources.ResultInfoGridFall, result.StartPosition, Math.Abs(climb));
                }

                return result.StartPosition.ToString();
            }

            return String.Empty;
        }

        public static string GetRaceTime(Result result) {
            if (result.IsQualifying) {
                return result.BestLapTime.ToLapTimeString();
            }

            Result winner = result.Replay.Winner;

            if (result == winner) {
                return result.RaceTime.ToLapTimeString();
            }
            else if (winner.LapsDone == result.LapsDone) {
                return String.Format(StringResources.ResultInfoTimeGap, (result.RaceTime - result.Replay.Winner.RaceTime).ToLapTimeString());
            }

            return String.Format(StringResources.ResultInfoTimeLaps, winner.LapsDone - result.LapsDone);
        }

        public static string GetPenalty(Result result) {
            if (result.ConfirmFlags.HasFlag(ConfirmFlags.DidNotPit)) {
                return StringResources.ResultInfoPenaltyPitStop;
            }

            if (result.ConfirmFlags.HasFlag(ConfirmFlags.DriveThrough)) {
                return StringResources.ResultInfoPenaltyDriveThrough;
            }

            if (result.ConfirmFlags.HasFlag(ConfirmFlags.StopGo)) {
                return StringResources.ResultInfoPenaltyStopGo;
            }

            if (result.PenaltyTime.Value.TotalSeconds > 0) {
                return String.Format(StringResources.ResultInfoPenaltyTime, result.PenaltyTime.Value.TotalSeconds);
            }

            return String.Empty;
        }

        public static string GetBestLap(Result result) {
            if (result.IsQualifying) {
                return String.Empty;
            }

            return result.BestLapTime.ToLapTimeString();
        }

        public static string GetPoints(Result result) {
            int points = result.Points;
            if (points > 0) {
                return points.ToString();
            }
            return String.Empty;
        }
    }
}
