﻿using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.ComponentModel;
using System.IO;
using System.Linq;
using System.Windows.Forms;
using LfsPoints.Commands;
using LfsPoints.Projects;
using LfsPoints.Properties;

namespace LfsPoints {
    public partial class Workbench : Form, IWorkbench {
        public Allocation ActiveAllocation {
            get { return (Allocation)_allocationsTabControl.SelectedTab.Tag; }
        }

        public Replay ActiveReplay {
            get { return _replaysListView.SelectedItem == null ? null : (Replay)_replaysListView.SelectedItem.Tag; }
        }

        public IEnumerable<Replay> ActiveReplays {
            get {
                foreach (ListViewItem item in _replaysListView.SelectedItems) {
                    yield return (Replay)item.Tag;
                }
            }
        }

        public Result ActiveResult {
            get { return _resultsListView.SelectedItem == null ? null : (Result)_resultsListView.SelectedItem.Tag; }
        }

        public IEnumerable<Result> ActiveResults {
            get {
                foreach (ListViewItem item in _resultsListView.SelectedItems) {
                    yield return (Result)item.Tag;
                }
            }
        }

        public Standing ActiveStanding {
            get { return _standingsListView.SelectedItem == null ? null : (Standing)_standingsListView.SelectedItem.Tag; }
        }

        public string ActiveUserName {
            get {
                if (ActiveResult != null) {
                    return ActiveResult.UserName;
                }
                if (ActiveStanding != null) {
                    return ActiveStanding.UserName;
                }
                return null;
            }
        }

        public Workbench(IEnumerable<string> commandLineArgs) {
            InitializeComponent();

            if (!HandleCommandLineArgs(commandLineArgs)) {
                NewProject();
            }

            _replaysListView.ColumnSorter = new ReplaysColumnSorter();
            _resultsListView.ColumnSorter = new ResultsColumnSorter();
            _standingsListView.ColumnSorter = new StandingsColumnSorter();
        }

        public bool HandleCommandLineArgs(IEnumerable<string> commandLineArgs) {
            if (commandLineArgs.Any()) {
                string filename = commandLineArgs.First();
                if (File.Exists(filename)) {
                    if (CanDestroyProject()) {
                        OpenProject(filename);

                        return true;
                    }
                }
                else {
                    DialogHelper.ShowError(
                        StringResources.InvalidCommandLineArgsDialogTitle,
                        StringResources.InvalidCommandLineArgsDialogMessage,
                        filename);
                }
            }

            return false;
        }

        public bool ExecuteCommand(ICommand command) {
            if (command.Execute(this)) {
                Project.Current.IsDirty = true;

                UpdateWorkbenchTitle();

                return true;
            }

            return false;
        }

        private void UpdateWorkbenchTitle() {
            if (Project.Current.IsDirty) {
                Text = String.Format(StringResources.WorkbenchTitleDirty, Project.Current.Name);
            }
            else {
                Text = String.Format(StringResources.WorkbenchTitle, Project.Current.Name);
            }
        }

        public void ClearWorkbench() {
            ClearAllocations();
            ClearReplays();
            ClearResults();
            ClearStandings();
        }

        public void NewProject() {
            if (CanDestroyProject()) {
                Project project = new Project(StringResources.DefaultProjectName);
                project.Allocations.Add(new Allocation(StringResources.DefaultAllocationName));
                Project.Current = project;

                ClearWorkbench();
                UpdateWorkbenchTitle();
                AddAllocations(project.Allocations);
            }
        }

        public void OpenProject() {
            if (CanDestroyProject()) {
                using (OpenFileDialog dialog = new OpenFileDialog()) {
                    dialog.Title = StringResources.OpenProjectDialogTitle;
                    dialog.Filter = StringResources.ProjectFileFilter;
                    dialog.AddExtension = true;
                    dialog.CheckFileExists = true;
                    dialog.InitialDirectory = Settings.Default.SaveDirectory;

                    if (DialogHelper.ShowDialog(dialog)) {
                        OpenProject(dialog.FileName);
                    }
                }
            }
        }

        public void OpenRecentProject(string fileName) {
            if (File.Exists(fileName)) {
                if (CanDestroyProject()) {
                    OpenProject(fileName);
                }
            }
            else if (DialogHelper.ShowYesNo(
                StringResources.RecentProjectMissingDialogTitle,
                StringResources.RecentProjectMissingDialogMessage,
                fileName)) {
                Settings.Default.RecentProjects.Remove(fileName);
            }
        }

        public void OpenProject(string fileName) {
            try {
                Cursor = Cursors.WaitCursor;

                Project.Current = Project.Load(fileName);
                AddProjectToSettings(fileName);

                ClearWorkbench();
                UpdateWorkbenchTitle();
                AddAllocations(Project.Current.Allocations);
                AddReplays(Project.Current.Replays);
                AddStandings();
            }
            catch (Exception ex) {
                DialogHelper.ShowError(
                    StringResources.OpenProjectErrorDialogTitle,
                    StringResources.OpenProjectErrorDialogMessage,
                    ex.Message);
            }
            finally {
                Cursor = null;
            }
        }

        public bool SaveProject() {
            if (Project.Current.IsSavable) {
                SaveProject(Project.Current.FileName);

                return true;
            }
            return SaveProjectAs();
        }

        public bool SaveProjectAs() {
            using (SaveFileDialog dialog = new SaveFileDialog()) {
                dialog.Title = StringResources.SaveProjectDialogTitle;
                dialog.Filter = StringResources.ProjectFileFilter;
                dialog.AddExtension = true;
                dialog.OverwritePrompt = true;
                dialog.FileName = Project.Current.Name;
                dialog.InitialDirectory = Project.Current.Directory ?? Settings.Default.SaveDirectory;

                if (DialogHelper.ShowDialog(dialog)) {
                    SaveProject(dialog.FileName);

                    return true;
                }

                return false;
            }
        }

        private void SaveProject(string fileName) {
            try {
                Cursor = Cursors.WaitCursor;

                Project.Save(fileName);

                AddProjectToSettings(fileName);

                UpdateWorkbenchTitle();
            }
            catch (Exception ex) {
                DialogHelper.ShowError(
                    StringResources.SaveProjectErrorDialogTitle,
                    StringResources.SaveProjectErrorDialogMessage,
                    ex.Message);
            }
            finally {
                Cursor = null;
            }
        }

        private bool CanDestroyProject() {
            if (Project.Current != null && Project.Current.IsDirty) {
                DialogResult result = DialogHelper.ShowYesNoCancel(
                    StringResources.SaveProjectDialogTitle,
                    StringResources.SaveProjectDialogMessage,
                    Project.Current.Name);

                if (result == DialogResult.Yes) {
                    return SaveProject();
                }
                else if (result == DialogResult.Cancel) {
                    return false;
                }
            }

            return true;
        }

        private static void AddProjectToSettings(string fileName) {
            const int MaxRecentProjects = 10;

            Settings.Default.SaveDirectory = Path.GetDirectoryName(fileName);

            StringCollection projects = Settings.Default.RecentProjects;
            if (projects.Contains(fileName)) {
                // Move project to top of list.
                projects.Remove(fileName);
                projects.Insert(0, fileName);
            }
            else {
                projects.Insert(0, fileName);

                if (projects.Count > MaxRecentProjects) {
                    projects.RemoveAt(projects.Count - 1);
                }
            }
        }

        protected override void OnClosing(CancelEventArgs e) {
            e.Cancel = !CanDestroyProject();

            base.OnClosing(e);
        }

        #region Allocations
        public void AddAllocations(IEnumerable<Allocation> allocations) {
            foreach (Allocation allocation in allocations) {
                AddAllocation(allocation);
            }
        }

        public void AddAllocation(Allocation allocation) {
            AddAllocation(allocation, false);
        }

        public void AddAllocation(Allocation allocation, bool selected) {
            AllocationTab tab = new AllocationTab(allocation);
            _allocationsTabControl.TabPages.Add(tab);

            if (selected) {
                _allocationsTabControl.SelectedTab = tab;
            }

            UpdateAllocationButtons();
        }

        public void RemoveAllocation(Allocation allocation) {
            _allocationsTabControl.TabPages.Remove((TabPage)allocation.Tag);

            UpdateAllocationButtons();
        }

        public void UpdateAllocation(Allocation allocation) {
            ((TabPage)allocation.Tag).Text = allocation.Name;
        }

        public void ClearAllocations() {
            _allocationsTabControl.TabPages.Clear();
        }

        private void UpdateAllocationButtons() {
            bool selected = _allocationsTabControl.SelectedTab != null;
            _removeAllocationButton.Enabled = selected;
            _renameAllocationButton.Enabled = selected;
            _exportAllocationButton.Enabled = selected;
        }

        private void _addAllocationButton_Click(object sender, EventArgs e) {
            ExecuteCommand(new AddAllocationCommand());
        }

        private void _removeAllocationButton_Click(object sender, EventArgs e) {
            ExecuteCommand(new RemoveAllocationCommand());
        }

        private void _renameAllocationButton_Click(object sender, EventArgs e) {
            ExecuteCommand(new RenameAllocationCommand());
        }

        private void _exportAllocationButton_Click(object sender, EventArgs e) {
            ExecuteCommand(new ExportAllocationCommand());
        }

        private void _importAllocationButton_Click(object sender, EventArgs e) {
            ExecuteCommand(new ImportAllocationCommand());
        }
        #endregion

        #region Replays
        public void AddReplays(IEnumerable<Replay> replays) {
            List<ListViewItem> items = new List<ListViewItem>();
            foreach (Replay replay in replays) {
                items.Add(GetReplayItem(replay));
            }
            _replaysListView.Items.AddRange(items.ToArray());
        }

        public void AddReplay(Replay replay) {
            AddReplay(replay, false);
        }

        public void AddReplay(Replay replay, bool selected) {
            ListViewItem item = GetReplayItem(replay);
            _replaysListView.Items.Add(item);

            if (selected) {
                item.Selected = true;
                item.EnsureVisible();
            }
        }

        public void RemoveReplays(IEnumerable<Replay> replays) {
            _replaysListView.BeginUpdate();
            foreach (Replay replay in replays) {
                _replaysListView.Items.Remove((ListViewItem)replay.Tag);
            }
            _replaysListView.EndUpdate();
        }

        public void UpdateReplays(IEnumerable<Replay> replays) {
            _replaysListView.BeginUpdate();
            foreach (Replay replay in replays) {
                UpdateReplayItem(replay);
            }
            _replaysListView.Sort();
            _replaysListView.EndUpdate();
        }

        public void UpdateReplay(Replay replay) {
            UpdateReplayItem(replay);

            _replaysListView.Sort();
        }

        private void UpdateReplayItem(Replay replay) {
            ListViewItem item = (ListViewItem)replay.Tag;
            _replaysListView.BeginUpdate();
            item.SubItems[_replayNameColumnHeader.Index].Text = replay.Name;
            item.SubItems[_replayResultsColumnHeader.Index].Text = replay.Results.Count.ToString();
            item.SubItems[_replayAllocationColumnHeader.Index].Text = replay.Allocation.Name;
            _replaysListView.EndUpdate();
        }

        public void ClearReplays() {
            _replaysListView.ClearItems();
        }

        private void _replaysListView_AfterLabelEdit(object sender, LabelEditEventArgs e) {
            if (_replaysListView.SelectedItem == null) {
                e.CancelEdit = true;
            }
            else {
                Replay replay = (Replay)_replaysListView.SelectedItem.Tag;
                if (String.IsNullOrWhiteSpace(e.Label) || e.Label == replay.Name) {
                    e.CancelEdit = true;
                }
                else {
                    ExecuteCommand(new RenameReplayCommand(e.Label));
                }
            }
        }

        private void _replaysListView_KeyDown(object sender, KeyEventArgs e) {
            if (_replaysListView.SelectedItem != null) {
                if (e.KeyCode == Keys.Delete) {
                    ExecuteCommand(new RemoveReplaysCommand());
                }

                if (e.KeyCode == Keys.F2) {
                    _replaysListView.SelectedItem.BeginEdit();
                }
            }
        }

        private static ListViewItem GetReplayItem(Replay replay) {
            ListViewItem item = new ListViewItem();
            item.Text = replay.Name;
            item.Tag = replay;
            replay.Tag = item;
            item.SubItems.AddRange(new string[] {
                ReplayHelper.GetTrack(replay),
                ReplayHelper.GetLaps(replay),
                replay.Results.Count.ToString(),
                replay.StartTime.ToString("g"),
                ReplayHelper.GetRaceMode(replay),
                replay.Allocation.Name,
            });
            return item;
        }

        private void _replaysListView_ItemSelected(object sender, EventArgs e) {
            Replay replay = (Replay)_replaysListView.SelectedItem.Tag;

            List<ListViewItem> items = new List<ListViewItem>();
            foreach (Result result in replay.Results) {
                items.Add(GetResultItem(result));
            }

            _resultsListView.Items.AddRange(items.ToArray());
        }

        private void _replaysListView_ItemDeselected(object sender, EventArgs e) {
            _resultsListView.ClearItems();
        }

        private static ListViewItem GetResultItem(Result result) {
            ListViewItem item = new ListViewItem();
            item.Text = result.Position.ToString();
            item.Tag = result;
            result.Tag = item;
            item.SubItems.AddRange(new string[] {
                result.DisplayName,
                ReplayHelper.GetCar(result),
                ReplayHelper.GetGrid(result),
                ReplayHelper.GetRaceTime(result),
                ReplayHelper.GetPenalty(result),
                ReplayHelper.GetBestLap(result),
                ReplayHelper.GetPoints(result),
            });
            return item;
        }

        public void UpdateResults(Replay replay) {
            UpdateResults(replay.Results);
        }

        public void UpdateResults(IEnumerable<Result> results) {
            _resultsListView.BeginUpdate();
            foreach (Result result in results) {
                ListViewItem item = (ListViewItem)result.Tag;
                item.SubItems[_resultPositionColumnHeader.Index].Text = result.Position.ToString();
                item.SubItems[_resultDriverColumnHeader.Index].Text = result.DisplayName;
                item.SubItems[_resultGridColumnHeader.Index].Text = ReplayHelper.GetGrid(result);
                item.SubItems[_resultTimeColumnHeader.Index].Text = ReplayHelper.GetRaceTime(result);
                item.SubItems[_resultPenaltyColumnHeader.Index].Text = ReplayHelper.GetPenalty(result);
                item.SubItems[_resultPointsColumnHeader.Index].Text = ReplayHelper.GetPoints(result);
            }
            _resultsListView.Sort();
            _resultsListView.EndUpdate();
        }

        public void RemoveResults(IEnumerable<Result> results) {
            _resultsListView.BeginUpdate();
            foreach (Result result in results) {
                _resultsListView.Items.Remove((ListViewItem)result.Tag);
            }
            _resultsListView.EndUpdate();
        }

        public void ClearResults() {
            _resultsListView.ClearItems();
        }

        private void _resultsListView_KeyDown(object sender, KeyEventArgs e) {
            if (_replaysListView.SelectedItem != null) {
                if (e.KeyCode == Keys.Delete) {
                    ExecuteCommand(new RemoveResultsCommand());
                }

                if (e.KeyCode == Keys.F2) {
                    Result result = ActiveResult;
                    ExecuteCommand(new CustomizeDisplayNameCommand(result.UserName, result.DisplayName));
                }
            }
        }
        #endregion

        #region Standings
        private Dictionary<string, ListViewItem> _standingItems = new Dictionary<string, ListViewItem>();

        public void AddStandings() {
            IEnumerable<Standing> standings = Project.Current.Standings;
            _standingItems.Clear();

            List<ListViewItem> items = new List<ListViewItem>();
            foreach (Standing standing in standings) {
                ListViewItem item = GetStandingItem(standing);
                items.Add(item);
                _standingItems[standing.UserName] = item;
            }

            _standingsListView.Items.Clear();
            _standingsListView.Items.AddRange(items.ToArray());
        }

        public void UpdateStandings() {
            IEnumerable<Standing> standings = Project.Current.Standings;

            _standingsListView.BeginUpdate();
            foreach (Standing standing in standings) {
                ListViewItem item = _standingItems[standing.UserName];
                item.SubItems[_standingPositionColumnHeader.Index].Text = standing.Position.ToString();
                item.SubItems[_standingPointsColumnHeader.Index].Text = standing.Points.ToString();
                item.Tag = standing;
            }
            _standingsListView.Sort();
            _standingsListView.EndUpdate();
        }

        public void UpdateStandingDisplayName(string userName, string displayName) {
            _standingItems[userName].SubItems[_standingDriverColumnHeader.Index].Text = displayName;
        }

        private static ListViewItem GetStandingItem(Standing standing) {
            ListViewItem item = new ListViewItem();
            item.Text = standing.Position.ToString();
            item.Tag = standing;
            standing.Tag = item;
            item.SubItems.AddRange(new string[] {
                standing.DisplayName,
                standing.Points.ToString(),
            });
            return item;
        }

        public void ClearStandings() {
            _standingsListView.ClearItems();
        }

        private void _standingsListView_KeyDown(object sender, KeyEventArgs e) {
            Standing standing = ActiveStanding;
            if (standing != null) {
                if (e.KeyCode == Keys.F2) {
                    ExecuteCommand(new CustomizeDisplayNameCommand(standing.UserName, standing.DisplayName));
                }
            }
        }
        #endregion

        public void ShowAbout() {
            using (AboutDialog dialog = new AboutDialog()) {
                DialogHelper.ShowDialog(dialog);
            }
        }
    }
}
