// LFS Vertex Shader : Car2


// Vertex shader input structure

struct VS_INPUT_PRELIT // VERT_PDST1
{
	float4	v0		: POSITION;
	float4	LitCol	: COLOR0;
	float4	AmbCol	: COLOR1;
	float2	v8		: TEXCOORD0;
};

struct VS_INPUT
{
	float4	v0		: POSITION;
	float3	Normal	: NORMAL;
	float4	Col		: COLOR;
	float2	v8		: TEXCOORD0;
};


// Vertex shader output structures

struct VS_OUTPUT_PRELIT
{
    float4	oPos		: POSITION;
	float	oFog		: FOG;
	float4	VCol		: COLOR0;
	float2	oT0			: TEXCOORD0;
};

struct VS_OUTPUT
{
    float4	oPos		: POSITION;
	float	oFog		: FOG;

	float2	oT0			: TEXCOORD0;
	float2	oT1			: TEXCOORD1;

	float3	VCol		: COLOR0;		// vertex colour
	float3	AmbientL	: COLOR1;		// ambient light colour
	float	DirectL		: TEXCOORD2;	// direct light amount

#ifdef SHINY
	float3	NormalEnv	: TEXCOORD3;
	float3	EyeToVEnv	: TEXCOORD4;
#endif

};

struct VS_OUTPUT_TINT
{
    float4	oPos		: POSITION;
	float	oFog		: FOG;

	float2	oT0			: TEXCOORD0;

	float3	VCol		: COLOR0;		// vertex colour
};


// Global variables

float4x4	lightinfo_mat	: register(c0);
//
float4		light_dir		: register(c8);
float4		up_dir			: register(c9);
float4		basecolamb		: register(c10);
float4		diffcolamb		: register(c11);
float4x4	local_to_env	: register(c12);
float4		eye_local		: register(c16);
//
float4x4	local_to_texel	: register(c20);
float4		clampmin		: register(c24);
float4		clampmax		: register(c25);

float4		fog_info		: register(c90);


// Main functions

float3 GammaToLinear(float3 col)
{
	return float3(pow(col.rgb, 2.2));
}


float3 LinearToGamma(float3 col)
{
	return float3(pow(col.rgb, 1/2.2));
}

float ComputeVolumetricFog(in float3 cameraToWorldPos, in float cameraHeight, in float density = 0.0003f)
{
	float cHeightFalloff = 0.01f;
	float cGlobalDensity = density;
	float cVolFogHeightDensityAtViewer = exp(-cHeightFalloff * cameraHeight);
	float fogInt = length(cameraToWorldPos) * cVolFogHeightDensityAtViewer;
	const float cSlopeThreshold = 0.01;
	if (abs(cameraToWorldPos.z) > cSlopeThreshold)
	{
		float t = cHeightFalloff * cameraToWorldPos.z;
		fogInt *= (1.0 - exp(-t)) / t;
	}
	return exp(-cGlobalDensity * fogInt);
}


float ComputeWorldSpaceFog(float3 vertPos)
{
	// Out.oPos.z * fog_info.x + fog_info.y; // Base LFS fog

	float3 V = mul(float4(vertPos - eye_local.xyz, 1.0f), local_to_env).xyz;
	return ComputeVolumetricFog(V, eye_local.z, fog_info.y / 2000);
}


// VSHADER_PRELIT_CAR
VS_OUTPUT_PRELIT vs_prelit_car( in VS_INPUT_PRELIT In ) // for prelit matt vertices in car space, e.g. lights / dashboards / mirrors
{
    VS_OUTPUT_PRELIT Out;
	Out.oPos = mul(In.v0, lightinfo_mat);
	Out.oFog = ComputeWorldSpaceFog(Out.oPos.xyz);
	Out.VCol = In.LitCol;
	Out.oT0 = In.v8;
    return Out;
}

// VSHADER_CAR_SOLID_SHINY
// VSHADER_CAR_SOLID_MATT
// VSHADER_CAR_ALPHA_SHINY
// VSHADER_CAR_ALPHA_MATT
VS_OUTPUT vs_main( in VS_INPUT In )
{
    VS_OUTPUT Out;
	Out.oPos = mul(In.v0, lightinfo_mat);
	Out.oFog = ComputeWorldSpaceFog(Out.oPos.xyz);
	Out.oT0 = In.v8;
	Out.oT1 = clamp(mul(In.v0, local_to_texel), clampmin, clampmax).xy; // lightmap texture

	Out.VCol = In.Col.rgb/2; // vertex colour
	
	// ambient lighting
	float up_dot_n = dot(In.Normal, up_dir.xyz);
	Out.AmbientL = basecolamb.rgb + up_dot_n * diffcolamb.rgb;
	//Out.AmbientL = basecolamb.rgb;
	
	// direct lighting
	Out.DirectL = max(0.0f, dot(In.Normal, light_dir.xyz));
	
#ifdef SHINY

#ifdef ALPHA
	// reflectivity of glass (parallel to normal)
	const float facing_reflectivity = 0.08f;
#else
	// reflectivity of paint (parallel to normal)
	const float facing_reflectivity = 0.04f;
#endif

	// reduce diffuse lighting when lit at glancing angles because more of the sun's light is reflected

	// Shlick approximation
	float schlick = pow(1.0f - Out.DirectL, 5);
	float direct_mul = lerp(1.0f - facing_reflectivity, 0.0f, schlick);

	Out.DirectL *= direct_mul;
	
	// for environment map, send the eye to vertex vector and the vertex normal, in environment map space
	Out.EyeToVEnv = mul(float4(In.v0.xyz - eye_local.xyz, 1.0f), local_to_env).xyz;
	Out.NormalEnv = mul(float4(In.Normal, 1.0f), local_to_env).xyz;
	
#endif // SHINY

//	Out.DirectL = 0.0f;							// ZERO direct lighting
//	Out.AmbientL = float3(0.0f, 0.0f, 0.0f);	// ZERO ambient lighting

    return Out;
}

// VSHADER_CAR_ALPHA_TINT
VS_OUTPUT_TINT vs_tint( in VS_INPUT In ) // tinted glass - currently unused
{
    VS_OUTPUT_TINT Out;

	// Pos and Fog
	Out.oPos = mul(In.v0, lightinfo_mat);
	Out.oFog = Out.oPos.z * fog_info.x + fog_info.y;

	Out.oT0 = In.v8; // diffuse texture coordinates

	Out.VCol = In.Col.rgb; // vertex colour

    return Out;
}