/* ------------------------------------------------------------------------- *
 * Copyright (C) 2007,2008 Dino Miniutti
 *
 * Dino Miniutti <dino [dot] miniutti [at] gmail [dot] com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * ------------------------------------------------------------------------- */
using System;
using System.IO;
using System.Collections.Generic;
using System.Text;
using FullMotion.LiveForSpeed.InSim;
using FullMotion.LiveForSpeed.InSim.EventHandlers;
using FullMotion.LiveForSpeed.InSim.Events;

namespace PenaltyReset
{
    public class PenaltyResetTracker
    {
        #region Member variables ######################################################################

        private static ConfigurationOptions cfg;
        private static InSimHandler handler;

        // state variables to make sure we don't start and stop the handler multiple times
        bool started = false;
        bool running = false;

        // Keep track of player names, since race tracking only gives us IDs
        Dictionary<byte, string> players = new Dictionary<byte, string>(); // <PlayerID,Name>
        Dictionary<byte, string> connections = new Dictionary<byte, string>(); // <ConnectionID,Name>
        // Also keep track of:
        Dictionary<byte, uint> playerLongPenaltyCounts = new Dictionary<byte, uint>(); // Player long (>10s) penalty count
        Dictionary<byte, uint> playerSGPenaltyCounts = new Dictionary<byte, uint>(); // Player normal (10s) penalty count
        Dictionary<byte, uint> playerDTPenaltyCounts = new Dictionary<byte, uint>(); // Player drive through penalty count
        Dictionary<byte, uint> playerStopTimes = new Dictionary<byte, uint>(); // Time of player's last stop

        #endregion

        #region Messaging & Logging Functions #########################################################

        // Create instructional buttons for the player when entering pits with a long stop-go penalty
        void SendPenaltyInstructionButtons(string DurationMessage, byte ConnectionId)
        {
            // Create the buttons
            LfsButton btn = new LfsButton(0);
            btn.Left = 25;
            btn.Top = 26;
            btn.Width = 150;
            btn.Height = 13;
            btn.Color = FullMotion.LiveForSpeed.InSim.Enums.ButtonColor.Dark;
            btn.Text = "** DO NOT GO WHEN YOU SEE [FINISHED] APPEAR **";

            LfsButton btn2 = new LfsButton(1);
            btn2.Left = 50;
            btn2.Top = 65;
            btn2.Width = 100;
            btn2.Height = 10;
            btn2.Color = FullMotion.LiveForSpeed.InSim.Enums.ButtonColor.Dark;
            btn2.Text = DurationMessage;

            // Send the buttons to the player
            handler.ShowButton(btn, ConnectionId);
            handler.ShowButton(btn2, ConnectionId);
        }

        // Delete the instructional buttons that the player receives when entering pits with a long stop-go penalty
        void DeletePenaltyInstructionButtons(byte ConnectionId)
        {
            // It doesn't matter if the buttons don't exist
            handler.DeleteButton(0, ConnectionId);
            handler.DeleteButton(1, ConnectionId);
        }

        // Inform the player of their penalty status
        void SendPenaltyStatusMessage(byte PlayerId)
        {
            if (playerLongPenaltyCounts[PlayerId] > 0 || playerSGPenaltyCounts[PlayerId] > 0 || playerDTPenaltyCounts[PlayerId] > 0)
            {
                SendMessageToPlayer("Penalties to complete:", PlayerId);
                if (playerDTPenaltyCounts[PlayerId] > 0) // Drive-through
                    SendMessageToPlayer("Drive-through: " + playerDTPenaltyCounts[PlayerId].ToString(), PlayerId);
                if (playerSGPenaltyCounts[PlayerId] > 0) // Normal
                    SendMessageToPlayer("Normal (10s): " + playerSGPenaltyCounts[PlayerId].ToString(), PlayerId);
                if (playerLongPenaltyCounts[PlayerId] > 0) // Long
                    SendMessageToPlayer("Long (" + cfg.penaltyOptions.PenaltyDurationInSeconds + "s): " + playerLongPenaltyCounts[PlayerId].ToString(), PlayerId);
            }
        }

        // Sends an information message to all players connected to host
        void SendInfoMessage(string subject, string message)
        {
            // Log locally
            LogMessage("  ==> ALL: \"" + subject + ": " + message + "\"");

            // Send to all players
            foreach (KeyValuePair<byte, string> p in players)
            {
                SendMessageToPlayer(subject + ":" + cfg.messageOptions.MessageColour + " " + message, p.Key, false);
            }
        }

        // Overloaded function, defaults to logging all messages.
        // Use the one below if you don't want to log the message.
        void SendMessageToPlayer(string message, byte playerID)
        {
            SendMessageToPlayer(message, playerID, true);
        }

        void SendMessageToPlayer(string message, byte playerID, bool logMessage)
        {
            // Log locally
            if (logMessage)
                LogMessage("  ==> " + players[playerID] + ": \"" + message + "\"");

            // Send to player
            handler.SendMessageToPlayer(cfg.messageOptions.MessageColour + message, playerID);
        }

        void LogMessage(string message, params object[] args)
        {
            string str = StripLFSColourCodes(System.DateTime.Now.ToString() + " -- " + String.Format(message, args));

            if (cfg.loggingOptions.LogToConsole)
                Console.WriteLine(str);

            if (cfg.loggingOptions.LogToFile)
            {
                StreamWriter w = File.AppendText(cfg.loggingOptions.LogFileName);
                w.WriteLine(str);
                w.Flush(); w.Close();
            }

        }

        // Remove LFS '^N'-style colour codes from a string
        string StripLFSColourCodes(string str)
        {
            // Remove colour codes
            for (int i = 0; i < 9; i++)
                str = str.Replace("^" + i.ToString(), "");

            return str;
        }

        #endregion

        #region InSim connection handling #############################################################

        // Connect to LFS and start tracking
        public void Start()
        {
            if (started)
            {
                throw new InvalidOperationException("RaceTracker cannot be started multiple times");
            }
            handler.Initialize();
            started = true;
            running = true;

            // Make sure we get all players in the race
            handler.RequestPlayerInfo();

            // Make sure we get all connections on the server
            handler.RequestConnectionInfo();
        }

        // Close down the connection
        public void Stop()
        {
            if (running)
            {
                handler.Close();
                running = false;
            }
        }

        // Start the race tracker
        public static void Run(ConfigurationOptions configurationOptions)
        {
            cfg = configurationOptions;

            PenaltyResetTracker tracker = new PenaltyResetTracker(cfg.connectionOptions.HostAddress,
                cfg.connectionOptions.Port,
                cfg.connectionOptions.AdminPassword);

            tracker.Start();
            Console.WriteLine("Tracker started. Press Enter to Exit...");
            Console.ReadLine();
            tracker.Stop();
        }

        // Set up an insim handler for tracking basic race info
        public PenaltyResetTracker(string host, int port, string adminpass)
        {
            handler = new InSimHandler();
            handler.Configuration.ProgramName = "PenaltyReset";
            handler.Configuration.LFSHost = host;
            handler.Configuration.LFSHostPort = port;
            handler.Configuration.AdminPass = adminpass;
            handler.Configuration.UseTCP = true;

            // Message handlers
            handler.RaceTrackPlayer += new RaceTrackPlayerHandler(handler_RaceTrackPlayer);
            handler.RaceTrackPlayerLeave += new RaceTrackPlayerLeaveHandler(handler_RaceTrackPlayerLeave);
            handler.RaceTrackCarReset += new RaceTrackCarResetHandler(hander_RaceTrackCarReset);
            handler.RaceTrackPlayerPenalty += new RaceTrackPlayerPenaltyHandler(handler_RaceTrackPlayerPenalty);
            handler.RaceTrackPlayerPitlane += new RaceTrackPlayerPitlaneHandler(handler_RaceTrackPlayerPitlane);
            handler.RaceTrackPlayerPitStopFinish += new RaceTrackPlayerPitStopFinishHandler(handler_RaceTrackPlayerPitStopFinish);
            handler.RaceTrackRaceStart += new RaceTrackRaceStartHandler(handler_RaceTrackRaceStart);
            handler.RaceTrackPlayerLap += new RaceTrackPlayerLapHandler(handler_RaceTrackPlayerLap);
            handler.RaceTrackCarTakeover += new RaceTrackCarTakeoverHandler(handler_RaceTrackCarTakeover);
            handler.RaceTrackConnection += new RaceTrackNewConnectionHandler(handler_RaceTrackConnection);
            handler.RaceTrackConnectionLeave += new RaceTrackConnectionLeaveHandler(handler_RaceTrackConnectionLeave);
            handler.RaceTrackPlayerRename += new RaceTrackPlayerRenameHandler(handler_RaceTrackPlayerRename);
        }

        #endregion

        #region Player & Connection tracking ##########################################################

        // New race starting
        void handler_RaceTrackRaceStart(InSimHandler sender, RaceTrackRaceStart e)
        {
            // Reset all penalty information at start of race
            foreach (KeyValuePair<byte, string> p in players)
            {
                playerLongPenaltyCounts[p.Key] = 0;
                playerSGPenaltyCounts[p.Key] = 0;
                playerDTPenaltyCounts[p.Key] = 0;
                playerStopTimes[p.Key] = 0;
            }

            // Delete any buttons that we sent to the players
            foreach (KeyValuePair<byte,string> con in connections)
                DeletePenaltyInstructionButtons(con.Key);

            LogMessage("***************************************************************");
            LogMessage("*** New race starting.");
        }

        // Player joined host
        private void handler_RaceTrackPlayer(InSimHandler sender, RaceTrackPlayer e)
        {
            if (!players.ContainsKey(e.PlayerId))
            {
                players.Add(e.PlayerId, e.Playername);
                playerLongPenaltyCounts.Add(e.PlayerId, 0);
                playerSGPenaltyCounts.Add(e.PlayerId, 0);
                playerDTPenaltyCounts.Add(e.PlayerId, 0);
                playerStopTimes.Add(e.PlayerId, 0);

                LogMessage("Player joined: {0}", e.Playername);
            }

            if (e.HandicapMass < 1)
            {
                handler.SendMessage("/spec " + players[e.PlayerId]);
                SendInfoMessage(players[e.PlayerId], "You must have a handicap");
            }
        }

        // Player left host
        private void handler_RaceTrackPlayerLeave(InSimHandler sender, RaceTrackPlayerLeave e)
        {
            LogMessage("Player left: {0}", players[e.PlayerId]);

            players.Remove(e.PlayerId);
            playerLongPenaltyCounts.Remove(e.PlayerId);
            playerSGPenaltyCounts.Remove(e.PlayerId);
            playerDTPenaltyCounts.Remove(e.PlayerId);
            playerStopTimes.Remove(e.PlayerId);
        }

        // Player taking over from another -- Issue penalties to new driver
        void handler_RaceTrackCarTakeover(InSimHandler sender, RaceTrackCarTakeover e)
        {
            LogMessage("Car takeover: {0} -> {1}", connections[e.OldConnectionId], connections[e.NewConnectionId]);

            players[e.PlayerId] = connections[e.NewConnectionId]; // Update player name
            SendPenaltyStatusMessage(e.PlayerId);
        }

        // Connection joined host
        void handler_RaceTrackConnection(InSimHandler sender, RaceTrackConnection e)
        {
            connections.Add(e.ConnectionId, e.Playername);
            LogMessage("Connection joined: {0}", connections[e.ConnectionId]);
        }
        
        // Connection left host
        void handler_RaceTrackConnectionLeave(InSimHandler sender, RaceTrackConnectionLeave e)
        {
            LogMessage("Connection left: {0}", connections[e.ConnectionId]);
            connections.Remove(e.ConnectionId);
        }

        // Connection changed name
        void handler_RaceTrackPlayerRename(InSimHandler sender, RaceTrackPlayerRename e)
        {
            string oldName = connections[e.ConnectionID];

            // Go through all the players in the list and change the old name to the new name
            foreach (KeyValuePair<byte, string> p in players)
            {
                if (p.Value == oldName)
                {
                    players[p.Key] = e.PlayerName;
                    break;
                }
            }

            // Rename the connection
            connections[e.ConnectionID] = e.PlayerName;

            LogMessage("Connection renamed: {0} -> {1}", oldName, e.PlayerName);
        }

        #endregion

        #region Race tracking #########################################################################

        // Player completed a lap (announce penalties)
        void handler_RaceTrackPlayerLap(InSimHandler sender, RaceTrackPlayerLap e)
        {
            if (cfg.messageOptions.AnnouncePenaltiesEveryLap)
                SendPenaltyStatusMessage(e.PlayerId);
        }

        // Player reset car
        private void hander_RaceTrackCarReset(InSimHandler sender, RaceTrackCarReset e)
        {
            LogMessage("Player reset: {0}", players[e.PlayerId]);
            handler.SendMessage("/p_sg " + players[e.PlayerId]); // Give player a stop-go penalty
            SendInfoMessage(players[e.PlayerId], "Long Stop-go penalty for resetting car.");

            playerLongPenaltyCounts[e.PlayerId]++;
        }


        // Player entering pitlane
        void handler_RaceTrackPlayerPitlane(InSimHandler sender, RaceTrackPlayerPitlane e)
        {
            if (e.EventType == FullMotion.LiveForSpeed.InSim.Enums.Pitlane.StopGo)
            {
                // Inform the user of their penalty status
                SendPenaltyStatusMessage(e.PlayerId);

                // Player entering pitlane for LONG or SHORT penalty
                if (playerSGPenaltyCounts[e.PlayerId] > 0 && playerLongPenaltyCounts[e.PlayerId] > 0)
                {
                    string DurationMessage = "You may stop for either 10s or " + cfg.penaltyOptions.PenaltyDurationInSeconds + "s.";
                    SendMessageToPlayer(DurationMessage, e.PlayerId);
                    SendPenaltyInstructionButtons(DurationMessage, GetConnectionIdFromPlayerId(e.PlayerId));
                }

                // Player entering pitlane for a LONG stop-go penalty only
                else if (playerLongPenaltyCounts[e.PlayerId] > 0)
                {
                    string DurationMessage = "You must stop for " + cfg.penaltyOptions.PenaltyDurationInSeconds + "s.";
                    SendMessageToPlayer(DurationMessage, e.PlayerId);
                    SendPenaltyInstructionButtons(DurationMessage, GetConnectionIdFromPlayerId(e.PlayerId));
                }

                // Player entering pitlane for SHORT penalty only
                else if (playerSGPenaltyCounts[e.PlayerId] > 0)
                    SendMessageToPlayer("You must stop for 10s.", e.PlayerId);
            }
        }


        // Player finished a pit-stop
        void handler_RaceTrackPlayerPitStopFinish(InSimHandler sender, RaceTrackPlayerPitStopFinish e)
        {
            LogMessage("Pit stop finish: {0}, Duration: {1}", players[e.PlayerId], e.StopTime.ToString());

            if (cfg.messageOptions.AnnounceStopTimeToPlayer)
                SendMessageToPlayer("You were stopped for: " + e.StopTime.ToString() + "ms.", e.PlayerId);

            // Save the duration of the stop
            playerStopTimes[e.PlayerId] = e.StopTime;

            // Delete the penalty instruction buttons
            DeletePenaltyInstructionButtons(GetConnectionIdFromPlayerId(e.PlayerId));
        }


        // Penalty: received/completed
        void handler_RaceTrackPlayerPenalty(InSimHandler sender, RaceTrackPlayerPenalty e)
        {
            // Player received a drive-through penalty
            if (e.NewPenalty == FullMotion.LiveForSpeed.InSim.Enums.Penalty.DriveThrough)
            {
                playerDTPenaltyCounts[e.PlayerId]++;
                SendPenaltyStatusMessage(e.PlayerId);
            }
            // Player completed a drive-through penalty
            if (e.NewPenalty == FullMotion.LiveForSpeed.InSim.Enums.Penalty.None &&
                e.OldPenalty == FullMotion.LiveForSpeed.InSim.Enums.Penalty.DriveThroughValid)
            {
                playerDTPenaltyCounts[e.PlayerId]--;
                SendPenaltyStatusMessage(e.PlayerId);
            }


            // Player received a LONG stop-go penalty from an admin (car reset)
            if (e.NewPenalty == FullMotion.LiveForSpeed.InSim.Enums.Penalty.StopGoValid
                && e.Reason == FullMotion.LiveForSpeed.InSim.Enums.PenaltyReason.Admin)
            {
                SendPenaltyStatusMessage(e.PlayerId);
            }
            // Player received a regular stop-go penalty for doing something in LFS (not admin penalty)
            else if (e.NewPenalty == FullMotion.LiveForSpeed.InSim.Enums.Penalty.StopGo
                && e.Reason != FullMotion.LiveForSpeed.InSim.Enums.PenaltyReason.StopShort)
            {
                playerSGPenaltyCounts[e.PlayerId]++; // Add the penalty to the list
                SendPenaltyStatusMessage(e.PlayerId);
            }

            // Player stopped to complete penalty
            else if (e.NewPenalty == FullMotion.LiveForSpeed.InSim.Enums.Penalty.None)
            {
                // Player has both long and normal penalties to complete
                if (playerSGPenaltyCounts[e.PlayerId] > 0 && playerLongPenaltyCounts[e.PlayerId] > 0)
                {
                    // Player completed long penalty
                    if (playerStopTimes[e.PlayerId] > cfg.penaltyOptions.PenaltyDurationInSeconds * 1000)
                    {
                        playerLongPenaltyCounts[e.PlayerId]--;
                    }
                    // Player completed short penalty
                    else if (playerStopTimes[e.PlayerId] > 10 * 1000)
                    {
                        playerSGPenaltyCounts[e.PlayerId]--;
                    }
                    // Player didn't stop long enough for either penalty
                    else
                    {
                        SendMessageToPlayer("You didn't stop for long enough.", e.PlayerId);
                    }
                }

                // Player has long penalty to complete only
                else if (playerLongPenaltyCounts[e.PlayerId] > 0)
                {
                    if(playerStopTimes[e.PlayerId] > cfg.penaltyOptions.PenaltyDurationInSeconds * 1000)
                        playerLongPenaltyCounts[e.PlayerId]--;
                    else
                        SendMessageToPlayer("You didn't stop for long enough.", e.PlayerId);
                }
                    
                // Player has hormal penalty to complete only
                else if (playerSGPenaltyCounts[e.PlayerId] > 0)
                {
                    if (playerStopTimes[e.PlayerId] > 10 * 1000)
                        playerSGPenaltyCounts[e.PlayerId]--;
                    else
                        SendMessageToPlayer("You didn't stop for long enough.", e.PlayerId);
                }


                // Player still has penalties remaining - tell LFS
                if (playerLongPenaltyCounts[e.PlayerId] > 0 || playerSGPenaltyCounts[e.PlayerId] > 0)
                {
                    handler.SendMessage("/p_sg " + players[e.PlayerId]);
                }
                else if (playerDTPenaltyCounts[e.PlayerId] > 0)
                {
                    handler.SendMessage("/p_dt " + players[e.PlayerId]);
                }
                else
                    SendMessageToPlayer("You don't have any more penalties to complete.", e.PlayerId);
            }
        }

        #endregion

        #region Utility Functions #####################################################################

        // Get the connection ID that is associated with the PlayerId
        byte GetConnectionIdFromPlayerId(byte PlayerId)
        {
            string PlayerName = players[PlayerId];

            foreach (KeyValuePair<byte, string> con in connections)
            {
                if (con.Value == PlayerName)
                    return con.Key;
            }

            return 0;
        }

        #endregion

    }
}