﻿/*
 *  Copyright Alex McBride 2008.
 * 
 *  This file is part of LFSReplayParser.
 *
 *  LFSReplayParser is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  LFSReplayParser is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with LFSReplayParser. If not, see <http://www.gnu.org/licenses/>.
 *  
 */

using System;
using System.IO;
using System.Text;
using System.Collections.Generic;

namespace LFSReplayParser.Extensions
{
    /// <summary>
    /// Extensions to the System.IO.BinaryReader class.
    /// </summary>
    internal static class BinaryReaderExtensions
    {
        static readonly Dictionary<char, Encoding> EncodingMap = new Dictionary<char, Encoding>()
            {
                {'L', Encoding.GetEncoding(1252)},
                {'G', Encoding.GetEncoding(1253)},
                {'C', Encoding.GetEncoding(1251)},
                {'J', Encoding.GetEncoding(932)},
                {'E', Encoding.GetEncoding(1250)},
                {'T', Encoding.GetEncoding(1254)},
                {'B', Encoding.GetEncoding(1257)},
                {'H', Encoding.GetEncoding(950)},
                {'S', Encoding.GetEncoding(936)},
                {'K', Encoding.GetEncoding(949)}
            };

        static readonly Dictionary<char, char> EscapeMap = new Dictionary<char, char>()
            {
                {'a', '*'},
                {'c', ':'},
                {'d', '\\'},
                {'s', '/'},
                {'q', '?'},
                {'t', '"'},
                {'l', '<'},
                {'r', '>'},
                {'^', '^'},
                {'v', '|'}
            };

        /// <summary>
        /// Move the position of the current stream by count bytes.
        /// </summary>
        /// <param name="br"></param>
        /// <param name="count">The number of bytes to skip.</param>
        public static void Skip(this BinaryReader br, int count)
        {
            br.ReadBytes(count);
        }

        /// <summary>
        ///  Read a string from the current stream and advance the position by count bytes.
        /// </summary>
        /// <param name="br"></param>
        /// <param name="count">The length of the string to read.</param>
        /// <returns></returns>
        public static string ReadString(this BinaryReader br, int count)
        {
            StringBuilder sb = new StringBuilder();
            foreach (char c in ASCIIEncoding.ASCII.GetString(br.ReadBytes(count)).ToCharArray())
            {
                if (c != char.MinValue)
                {
                    sb.Append(c);
                }
            }
            return sb.ToString();
        }

        /// <summary>
        /// Read a LFS encoded string from the current stream and advance the position by count bytes.
        /// </summary>
        /// <param name="br"></param>
        /// <param name="count">The length of the string to read</param>
        public static string ReadStringEncoded(this BinaryReader br, int count)
        {
            byte[] data = br.ReadBytes(count);
            StringBuilder output = new StringBuilder();
            List<byte> accum = new List<byte>();
            Encoding encoding = EncodingMap['L'];
            bool ctrlChar = false;

            foreach (byte b in data)
            {
                if (ctrlChar)
                {
                    if (EncodingMap.ContainsKey((char)b))
                    {
                        encoding = EncodingMap[(char)b];
                    }
                    else if (EscapeMap.ContainsKey((char)b))
                    {
                        accum.Add((byte)EscapeMap[(char)b]);
                    }
                    ctrlChar = false;
                }
                else if (b != byte.MinValue)
                {
                    if ('^' == (char)b)
                    {
                        ctrlChar = true;
                        output.Append(encoding.GetString(accum.ToArray()));
                        accum.Clear();
                    }
                    else
                    {
                        accum.Add(b);
                    }
                }
            }

            if (accum.Count > 0)
            {
                output.Append(encoding.GetString(accum.ToArray()));
            }

            return output.ToString();
        }

        /// <summary>
        /// Read a boolean from the current stream and advance the position by one byte.
        /// </summary>
        public static bool ReadByteAsBool(this BinaryReader br)
        {
            return (br.ReadByte() == 1);
        }

        /// <summary>
        /// Read a unix timestamp from the current stream and advance the position by four bytes.
        /// </summary>
         public static DateTime ReadTimeStamp(this BinaryReader br)
        {
            uint unixTime = br.ReadUInt32();
            return new DateTime(1970, 1, 1, 0, 0, 0).AddSeconds(unixTime).ToLocalTime();
        }
    }
}
