﻿/*
 *  Copyright Alex McBride 2008.
 * 
 *  This file is part of LFSReplayParser.
 *
 *  LFSReplayParser is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  LFSReplayParser is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with LFSReplayParser. If not, see <http://www.gnu.org/licenses/>.
 *  
 */

using System;
using System.IO;

using LFSReplayParser.Extensions;

namespace LFSReplayParser.Replays
{
    /// <summary>
    /// Represents a MultiplayerReplay result.
    /// </summary>
    public class Result : IComparable<Result>
    {
        /// <summary>
        /// The players final position in the race.
        /// </summary>
        public int  Position { get;private set; }

        /// <summary>
        /// The players name.
        /// </summary>
        public string PlayerName { get; private set; }

        /// <summary>
        /// The players number plate.
        /// </summary>
        public string NumberPlate { get; private set; }

        /// <summary>
        /// The name of the car the player is driving (EG FBM).
        /// </summary>
        public string ShortCarName { get; private set; }

        /// <summary>
        /// The players LFSWorld username.
        /// </summary>
        public string LfsUserName { get; private set; }

        /// <summary>
        /// The number of laps the player has completed.
        /// </summary>
        public int LapsDone { get; private set; }

        /// <summary>
        /// The players settings flags.
        /// </summary>
        public PlayerFlags PlayerFlags { get; private set; }

        /// <summary>
        /// The players race results confirmation flags.
        /// </summary>
        public ConfirmFlags ConfirmFlags { get; private set; }

        /// <summary>
        /// The number of pit stops the player completed.
        /// </summary>
        public int Pitstops { get; private set; }

        /// <summary>
        /// The number of seconds time penalty the player recieved.
        /// </summary>
        public int PenaltySeconds { get; private set; }

        /// <summary>
        /// The players overall race time.
        /// </summary>
        public Time OverallTime { get; private set; }

        /// <summary>
        /// The players best lap time.
        /// </summary>
        public Time BestLapTime { get; private set; }

        /// <summary>
        /// The players start position in the race, or zero if they were not given a start position (EG mid-race join).
        /// </summary>
        public int StartPosition { get; private set; }

        /// <summary>
        /// The players amount of handicap mass (KG).
        /// </summary>
        public int HandicapMass { get; private set; }

        /// <summary>
        /// The players intake restriction (%).
        /// </summary>
        public int IntakeRestriction { get; private set; }

        /// <summary>
        /// Create a new Result object.
        /// </summary>
        public Result() { }

        /// <summary>
        /// Compare two results, for use in sorting.
        /// </summary>
        /// <param name="result">The result to compare this one to.</param>
        /// <returns>An interger reprentation of the comparison.</returns>
        public int CompareTo(Result result)
        {
            return Position.CompareTo(result.Position);
        }

        /// <summary>
        /// Return a string representation of the Result object.
        /// </summary>
        /// <returns>The PlayerName.</returns>
        public override string ToString()
        {
            return PlayerName;
        } 

        /// <summary>
        /// Parse a Result object from file.
        /// </summary>
        /// <param name="br">The current BinaryReader to read the data from.</param>
        /// <param name="position">The players position in the race.</param>
        /// <returns>A Result object populated with result data.</returns>
        internal static Result Parse(BinaryReader br, int position)
        {
            Result result = new Result()
            {
                Position = position,
            };

            result.PlayerName = br.ReadStringEncoded(24);
            result.NumberPlate = br.ReadStringEncoded(8);
            result.ShortCarName = br.ReadString(4);
            result.LfsUserName = br.ReadStringEncoded(24);
            result.LapsDone = br.ReadInt16();
            result.PlayerFlags = (PlayerFlags)br.ReadInt16();
            result.ConfirmFlags = (ConfirmFlags)br.ReadByte();
            result.Pitstops = br.ReadByte();
            result.PenaltySeconds = br.ReadInt16();
            result.OverallTime = new Time(br.ReadInt32());
            result.BestLapTime = new Time(br.ReadInt32());
            br.Skip(1);
            result.StartPosition = br.ReadByte();
            result.HandicapMass = br.ReadByte();
            result.IntakeRestriction = br.ReadByte();
            return result;
        }
    }
}
