﻿/*
 *  Copyright Alex McBride 2008.
 * 
 *  This file is part of LFSReplayParser.
 *
 *  LFSReplayParser is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  LFSReplayParser is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with LFSReplayParser. If not, see <http://www.gnu.org/licenses/>.
 *  
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace LFSReplayParser.Replays
{
    /// <summary>
    /// Represents a time span in LFS.
    /// </summary>
    public class Time : IComparable<Time>
    {
        /// <summary>
        /// Get the total number of milliseconds comprising the time.
        /// </summary>
        public int TotalMilliseconds { get; private set; }

        /// <summary>
        /// Get the number of hours in the time.
        /// </summary>
        public int Hours
        {
            get
            {
                return (int)Math.Floor((double)TotalMilliseconds / 3600000);
            }
        }

        /// <summary>
        /// Get the number of minutes in the time.
        /// </summary>
        public int Minutes
        {
            get
            {
                return (int)Math.Floor((double)TotalMilliseconds / 1000 / 60);
            }
        }

        /// <summary>
        /// Get the number of seconds in the time.
        /// </summary>
        public int Seconds
        {
            get
            {
                return (int)Math.Floor((double)TotalMilliseconds / 1000) % 60;
            }
        }

        /// <summary>
        /// Get the number of hundredths in the time.
        /// </summary>
        public int Hundredths
        {
            get
            {
                return (int)TotalMilliseconds % 1000;
            }
        }

        /// <summary>
        /// Get if the Time object contains no value.
        /// </summary>
        public bool IsZero
        {
            get
            {
                return (TotalMilliseconds == 0);
            }
        }

        /// <summary>
        /// Create a new Time object.
        /// </summary>
        public Time() { }

        /// <summary>
        /// Create a new Time object.
        /// </summary>
        /// <param name="milliseconds">The total number of milliseconds in the time.</param>
        public Time(int milliseconds)
        {
            TotalMilliseconds = milliseconds;
        }

        /// <summary>
        /// Create a new Time object.
        /// </summary>
        /// <param name="minutes">The minutes of the time.</param>
        /// <param name="seconds">The seconds of the time.</param>
        /// <param name="hundredths">The hundredths of the time.</param>
        /// <param name="thousanths">The thousanths of the time.</param>
        public Time(int minutes, int seconds, int hundredths, int thousanths)
        {
            TotalMilliseconds = minutes * 60000;
            TotalMilliseconds += seconds * 1000;
            TotalMilliseconds += hundredths * 10;
            TotalMilliseconds += thousanths;
        }

        /// <summary>
        /// Add a time to the current Time object.
        /// </summary>
        /// <param name="time"></param>
        public Time Add(Time time)
        {
            return new Time(TotalMilliseconds + time.TotalMilliseconds);
        }

        /// <summary>
        /// Subtract a time from the current Time object.
        /// </summary>
        /// <param name="time"></param>
        public Time Subtract(Time time)
        {
            return new Time(TotalMilliseconds - time.TotalMilliseconds);
        }

        /// <summary>
        /// Compare this Time object with another, for use in sorting comparisons.
        /// </summary>
        /// <param name="time">The Time object to compare this one to.</param>
        /// <returns>The integer representation of the comparison.</returns>
        public int CompareTo(Time time)
        {
            return TotalMilliseconds.CompareTo(time.TotalMilliseconds);
        }

        /// <summary>
        /// Return a string representation of the Time.
        /// </summary>
        /// <returns>A string representation of the time.</returns>
        public override string ToString()
        {
            if (Hours > 0)
                return string.Format("{0}:{1:00}:{2:00}.{3:000}", Hours, Minutes, Seconds, Hundredths);
            else
                return string.Format("{0}:{1:00}.{2:000}", Minutes, Seconds, Hundredths);
        }
    }
}
