package lfstrackdatatesting;

import java.util.Arrays;
import java.util.HashMap;

/**
 * Represents one track from LFS. Contains all of the node information
 * found in the track's PTH file
 * @author Gus
 *
 */

public class TrackData {
	private String trackName;
	
	private HashMap<Integer, OneNode> trackNodes = new HashMap<Integer, OneNode>();
	
	/**
	 * Create a track with the given name
	 * @param trackName - the track name (e.g. AS5)
	 */
	public TrackData(String trackName) {
		this.trackName = trackName;
	}
	
	/**
	 * Add one node to this track
	 * @param node
	 */
	public void addOneNode(OneNode node) {
		trackNodes.put(node.getNodeIndex(), node);
	}
	
	/**
	 * Get an ordered array of all the nodes in this track.
	 * @return
	 */
	public OneNode[] getAllNodes() {
		if(trackNodes != null && trackNodes.size() > 0) {
			OneNode[] nodes = trackNodes.values().toArray(
					new OneNode[trackNodes.size()]);
			
			Arrays.sort(nodes);
			
			return nodes;
		} else {
			return null;
		}
	}
	
	/**
	 * Get the name of this track
	 * @return
	 */
	public String getTrackName() {
		return trackName;
	}
	
	/**
	 * Get the number of nodes in this track
	 * @return
	 */
	public int getNodeCount() {
		return trackNodes.size();
	}
	
	/*
	 * For testing / debugging purposes. Returns a big string containing all of the differences
	 * between 'neighbour' nodes.
	 */
	public String getDifferenceBetweenAllNodesString() {
		StringBuilder string = new StringBuilder();
		
		if(trackNodes != null && trackNodes.size() > 0) {
			for(int i=0; i<trackNodes.size(); i++) {
				if(i < trackNodes.size()-1) {
					string.append(trackNodes.get(i).getDifferenceComparedToAnotherNode(
							trackNodes.get(i+1))+"\n");
				}
			}
		}
		
		return string.toString();
	}
	
	/**
	 * Returns the length of this track in metres.
	 * Credit to avetere from here 
	 * http://www.lfsforum.net/showthread.php?p=1173222#post1173222
	 * @return
	 */
	public double getTrackLength() {
		double length = 0;
		
		for(int i=0; i<trackNodes.size(); i++) {
			if(i<trackNodes.size()-1) {
				OneNode thisNode = trackNodes.get(i);
				OneNode nextNode = trackNodes.get(i+1);
				
				double diffX = Math.pow((nextNode.getCentreX() - thisNode.getCentreX())/65536.0, 2);
				double diffY = Math.pow((nextNode.getCentreY() - thisNode.getCentreY())/65536.0, 2);
				double diffZ = Math.pow((nextNode.getCentreZ() - thisNode.getCentreZ())/65536.0, 2);
				
				length += Math.sqrt(diffX + diffY + diffZ);
			} else {
				OneNode thisNode = trackNodes.get(i);
				OneNode firstNode = trackNodes.get(0);
				
				double diffX = Math.pow((firstNode.getCentreX() - thisNode.getCentreX())/65536.0, 2);
				double diffY = Math.pow((firstNode.getCentreY() - thisNode.getCentreY())/65536.0, 2);
				double diffZ = Math.pow((firstNode.getCentreZ() - thisNode.getCentreZ())/65536.0, 2);

				length += Math.sqrt(diffX + diffY + diffZ);
			}
		}
		
		return length;
	}
}